<?php

declare(strict_types=1);

namespace Drupal\lms_file_upload\Plugin\ActivityAnswer;

use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\file\FileInterface;
use Drupal\file\Plugin\Field\FieldType\FileItem;
use Drupal\file\Validation\FileValidatorSettingsTrait;
use Drupal\lms\Attribute\ActivityAnswer;
use Drupal\lms\Entity\Answer;
use Drupal\lms\Plugin\ActivityAnswerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * File upload ActivityAnswer plugin.
 */
#[ActivityAnswer(
  id: 'file_upload',
  name: new TranslatableMarkup('File upload'),
)]
final class FileUpload extends ActivityAnswerBase implements PluginFormInterface {

  use FileValidatorSettingsTrait;

  /**
   * File URL generator service.
   */
  protected FileUrlGeneratorInterface $fileUrlGenerator;

  /**
   * {@inheritdoc}
   */
  public function injectServices(ContainerInterface $container): void {
    $this->fileUrlGenerator = $container->get(FileUrlGeneratorInterface::class);
    parent::injectServices($container);
  }

  /**
   * {@inheritdoc}
   *
   * @see \Drupal\file\Plugin\Field\FieldType\FileItem::fieldSettingsForm
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $settings = $this->configuration;

    $extensions = \str_replace(' ', ', ', $settings['default_file_extensions'] ?? 'txt');
    $form['default_file_extensions'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default allowed file extensions'),
      '#description' => $this->t("Separate extensions with a comma or space. Each extension can contain alphanumeric characters, '.', and '_', and should start and end with an alphanumeric character. <strong>This value can be overridden on a per-activity basis.</strong>"),
      '#required' => TRUE,
      '#default_value' => $extensions,
      '#element_validate' => [[FileItem::class, 'validateExtensions']],
      '#maxlength' => 256,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function answeringForm(array &$form, FormStateInterface $form_state, Answer $answer): void {
    $activity = $answer->getActivity();
    $data = $answer->getData();

    $default_value = $data['file_upload'][0] ?? NULL;
    $default_file_extensions = $this->configuration['default_file_extensions'];
    $file_extensions = $activity->get('field_file_extensions')->first()?->getValue()['value'] ?? $default_file_extensions;
    $file_extensions = \str_replace(',', ' ', $file_extensions);

    $form['file_upload'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('File'),
      '#description' => $this->t('Upload a file. Allowed extensions: @extensions.', ['@extensions' => $file_extensions]),
      '#required' => TRUE,
      '#default_value' => $default_value ? ['fids' => $default_value] : NULL,
      '#upload_location' => 'private://lms_file_upload',
      '#upload_validators' => $this->getFileUploadValidators(['file_extensions' => $file_extensions]),
    ];
  }

  /**
   * {@inheritdoc}
   */
  protected function getAnswerRenderable(Answer $answer): array {
    $data = $answer->getData();

    if (
      !\array_key_exists('file_upload', $data) ||
      !\is_array($data['file_upload']) ||
      !\array_key_exists(0, $data['file_upload'])
    ) {
      return ['#markup' => $this->t('No file uploaded')];
    }

    $file = $this->entityTypeManager->getStorage('file')->load($data['file_upload'][0]);
    if (!$file instanceof FileInterface) {
      return ['#markup' => $this->t('Uploaded file is missing')];
    }

    return [
      '#type' => 'link',
      '#title' => $file->getFilename(),
      '#attributes' => [
        'class' => ['lms-answer-file'],
        'target' => '_blank',
      ],
      '#url' => $this->fileUrlGenerator->generate($file->getFileUri()),
    ];
  }

  /**
   * Get default file extensions.
   */
  public function getDefaultAllowedFileExtensions(): string {
    return $this->configuration['default_file_extensions'];
  }

}
