<?php

declare(strict_types=1);

namespace Drupal\Tests\lms_file_upload\FunctionalJavascript;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Url;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Tests\lms\LmsTestHelperTrait;
use Drupal\lms\Entity\AnswerInterface;

/**
 * General javascript test of LMS basic features.
 *
 * @group lms
 */
final class LmsFileUploadTest extends WebDriverTestBase {

  use LmsTestHelperTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'dblog',
    'block',
    'page_cache',
    'dynamic_page_cache',
    'lms',
    'lms_classes',
    'lms_file_upload',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'claro';

  /**
   * Test user accounts.
   */
  private array $users = [];

  /**
   * Test data to pass between methods.
   */
  private array $testData = [];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Import test config.
    $this->importTestConfig();

    // Create test users.
    $users_source_file_path = $this->container->get(ModuleExtensionList::class)->getPath('lms') . '/tests/data/users.yml';
    $user_data = Yaml::decode(\file_get_contents($users_source_file_path));
    foreach ($user_data as $name => $values) {
      $this->users[$name] = $this->drupalCreateUser([], NULL, FALSE, $values);
    }
  }

  /**
   * Test router.
   */
  public function testLmsFileUpload(): void {
    $this->activityTypeTest();
    $this->courseCreationTest();
    $this->activityTakingTest();
    $this->courseResultsTest();
    $this->watchdogTest();
  }

  /**
   * Create LMS file upload activity type..
   */
  private function activityTypeTest(): void {
    $page = $this->getSession()->getPage();
    $assert_session = $this->assertSession();

    $this->drupalLogin($this->users['admin']);

    // Activity type creation test.
    $this->drupalGet(Url::fromRoute('entity.lms_activity_type.collection'));

    $page->clickLink('Add activity type');
    $this->setFormElementValue('input', 'name', 'File upload');
    $this->setFormElementValue('select', 'pluginId', 'file_upload');

    $assert_session->waitForElementVisible('css', '[name^="pluginConfiguration"]');
    $page->fillField('pluginConfiguration[default_file_extensions]', 'txt jpg');
    $this->pressButton('edit-submit');
    $this->screenShot();

    self::assertTrue(\strpos(\strip_tags($page->getHtml()), 'File upload') !== FALSE, 'The File upload activity type is not installed.');
  }

  /**
   * Course creation test.
   */
  private function courseCreationTest(): void {
    $this->drupalLogin($this->users['teacher']);

    // Create test activity.
    $activity_name = 'Test file upload activity';
    $this->drupalGet(Url::fromRoute('entity.lms_activity.add_form', [
      'lms_activity_type' => 'file_upload',
    ]));
    $this->setEntityFormField('name', $activity_name);
    $this->pressButton('edit-submit');
    $this->testData['activity'] = $this->getEntityByProperties('lms_activity', [
      'name' => $activity_name,
    ]);

    // Create a lesson.
    $lesson_name = 'Test file upload lesson';
    $this->drupalGet(Url::fromRoute('entity.lms_lesson.add_form', [
      'lms_lesson_type' => 'lesson',
    ]));
    $this->setEntityFormField('name', $lesson_name);
    $this->setLmsReferenceField('activities', [['entity' => $this->testData['activity']]]);
    $this->pressButton('edit-submit');
    $this->testData['lesson'] = $this->getEntityByProperties('lms_lesson', [
      'name' => $lesson_name,
    ]);

    // Create a course.
    $course_name = 'Test file upload course';
    $this->drupalGet(Url::fromRoute('entity.group.add_form', [
      'group_type' => 'lms_course',
    ]));
    $this->setEntityFormField('label', $course_name);
    $this->setLmsReferenceField('lessons', [['entity' => $this->testData['lesson']]]);
    $this->pressButton('edit-submit');
    $this->testData['course'] = $this->getEntityByProperties('group', [
      'label' => $course_name,
    ]);
  }

  /**
   * Activity take test.
   */
  private function activityTakingTest(): void {
    $page = $this->getSession()->getPage();

    $this->drupalLogin($this->users['student']);

    $this->drupalGet(Url::fromRoute('entity.group.canonical', [
      'group' => $this->testData['course']->id(),
    ]));
    $page->clickLink('Enroll');
    $this->pressButton('edit-submit');
    $page->clickLink('Start');

    $this->testData['filename'] = $this->container->get(FileSystemInterface::class)->tempnam('temporary://', "testManagedFile") . '.txt';
    \file_put_contents($this->testData['filename'], $this->randomString(128));
    $page->attachFileToField('files[file_upload]', $this->container->get('file_system')->realpath($this->testData['filename']));
    $remove_button = $this->assertSession()->waitForElementVisible('css', '[name="file_upload_remove_button"]');
    $this->assertNotNull($remove_button);

    $this->pressButton('[data-drupal-selector="edit-submit"]', 'css');
    $this->screenShot();
  }

  /**
   * Course results test.
   */
  private function courseResultsTest(): void {
    $page = $this->getSession()->getPage();

    $this->drupalLogin($this->users['teacher']);
    $student_id = $this->users['student']->id();

    // Results page.
    $this->drupalGet(Url::fromRoute('lms.group.results', [
      'group' => $this->testData['course']->id(),
      'user' => $student_id,
    ]));

    $page->find('css', '.lesson-score-details')->click();

    $answer_id = $this->getEntityIdByProperties('lms_answer', [
      // Skip lesson status condition to save some code - as long as
      // this activity is used only in a single lesson, all is fine.
      'user_id' => $student_id,
      AnswerInterface::ACTIVITY_FIELD => $this->testData['activity']->id(),
    ]);

    $details_uri = Url::fromRoute('lms.answer.details', [
      'lms_answer' => $answer_id,
      'js' => 'nojs',
    ])->toString();
    $this->getSession()->getPage()->find('css', '[href="' . $details_uri . '"]')->click();
    $modal = $this->assertSession()->waitForElementVisible('css', '[role="dialog"]');
    $link = $modal->find('css', 'a.lms-answer-file');
    $this->assertElementTextContains($link, \basename($this->testData['filename']), 'Uploaded file not found on evaluation modal.');
  }

}
