<?php

declare(strict_types=1);

namespace Drupal\lms_membership_request\Hook;

use Drupal\Core\Entity\ContentEntityFormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Link;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\group\Entity\Form\GroupRelationshipDeleteForm;
use Drupal\group\Entity\GroupRelationshipInterface;
use Drupal\lms\Entity\Bundle\Course;
use Drupal\lms_classes\ClassHelper;
use Drupal\lms_classes\Hook\LmsClassesMembershipHooksBase;

/**
 * Membership form alter hooks logic.
 */
final class CourseMembershipRequestHooks extends LmsClassesMembershipHooksBase {

  use StringTranslationTrait;
  use MessengerTrait;

  #[Hook('form_group_relationship_form_alter')]
  #[Hook('form_group_relationship_confirm_form_alter')]
  public function groupRelationshipFormAlter(&$form, FormStateInterface $form_state): void {
    $form_object = $form_state->getFormObject();
    if (
      !$form_object instanceof ContentEntityFormInterface ||
      $form_object instanceof GroupRelationshipDeleteForm
    ) {
      return;
    }

    $group_relationship = $form_object->getEntity();
    \assert($group_relationship instanceof GroupRelationshipInterface);
    $relationship_plugin_id = $group_relationship->getPluginId();
    if ($relationship_plugin_id !== 'group_membership_request') {
      return;
    }

    $course = $group_relationship instanceof GroupRelationshipInterface ? $group_relationship->getGroup() : NULL;
    if (!$course instanceof Course) {
      return;
    }

    // Common alters.
    $form_state->set('redirect_group_id', $course->id());
    $form['actions']['submit']['#submit'][] = [
      self::class,
      'redirectToCourse',
    ];
    $form['actions']['cancel'] = Link::createFromRoute($this->t('Cancel'),
      'entity.group.canonical',
      ['group' => $course->id()],
      ['attributes' => ['class' => 'button']]
    )->toRenderable();

    if ($group_relationship->isNew()) {
      $this->groupMembershipRequestFormAlter($form, $form_state, $course, $group_relationship);
    }
    else {
      $this->groupMembershipRequestApproveFormAlter($form, $form_state, $course, $group_relationship);
    }

  }

  /**
   * Group membership request form alter.
   */
  private function groupMembershipRequestFormAlter(
    array &$form,
    FormStateInterface $form_state,
    Course $course,
    GroupRelationshipInterface $group_relationship,
  ): void {
    // Check if class membership doesn't already exist if user navigated
    // here somehow.
    /** @var \Drupal\Core\Session\AccountInterface */
    $requesting_user = $group_relationship->getEntity();
    if (ClassHelper::isClassMember($requesting_user, $course)) {
      $form['info'] = [
        '#type' => 'container',
        '#attributes' => ['class' => 'lms-info'],
        'content' => [
          '#markup' => $this->t('You are already a member of a class on this course and can access it.'),
        ],
        '#weight' => -100,
      ];
      $form['actions'] = [
        'back' => Link::createFromRoute($this->t('Back'),
          'entity.group.canonical',
          ['group' => $course->id()],
          ['attributes' => ['class' => 'button']]
        )->toRenderable(),
      ];
      return;
    }

    $form['actions']['submit']['#value'] = $this->t('Request subscription');
  }

  /**
   * Group membership request approve form alter.
   */
  private function groupMembershipRequestApproveFormAlter(
    array &$form,
    FormStateInterface $form_state,
    Course $course,
    GroupRelationshipInterface $group_relationship,
  ): void {
    $form['#validate'][] = [
      $this,
      'membershipApproveFormValidate',
    ];

    $this->addClassSelector($form, $course, FALSE);

    if (\array_key_exists('group_roles', $form)) {
      $form['group_roles']['#states'] = [
        'visible' => [
          'select[name=class]' => ['value' => '0'],
        ],
      ];
    }
  }

  /**
   * Membership form validate handler.
   */
  public function membershipApproveFormValidate(array $form, FormStateInterface $form_state): void {
    /** @var \Drupal\Core\Entity\ContentEntityFormInterface */
    $form_object = $form_state->getFormObject();
    /** @var \Drupal\group\Entity\GroupRelationshipInterface */
    $group_relationship = $form_object->getEntity();

    // Roles selected - don't do anything.
    $roles = $form_state->getValue('roles');
    if (\is_array($roles) && \count(\array_filter($roles)) > 0) {
      return;
    }

    $class_id = $form_state->getValue('class');
    if ($class_id === NULL || $class_id === '0') {
      return;
    }

    // Since this is a group membership request, the only thing we need
    // to change is the target group ID.
    $group_relationship->set('gid', $class_id);
  }

}
