<?php

declare(strict_types=1);

namespace Drupal\Tests\lms\FunctionalJavascript;

use Drupal\Core\Url;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Tests\lms\LmsTestHelperTrait;

/**
 * General javascript test of LMS basic features.
 *
 * @group lms
 */
final class LmsMembershipTest extends WebDriverTestBase {

  use LmsTestHelperTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'claro';

  /**
   * Modules to install.
   *
   * @var array
   */
  protected static $modules = [
    'dblog',
    'block',
    'page_cache',
    'dynamic_page_cache',
    'lms_classes',
    'lms_answer_plugins',
    'lms_membership_request',
  ];

  /**
   * Test user data.
   */
  private array $userData;

  /**
   * Test Course data.
   */
  private array $courseData;

  /**
   * Test user accounts.
   */
  private array $users = [];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Set source data.
    $this->setSourceData();

    // Import test config.
    $this->importTestConfig();

    // Create test content.
    $this->createTestContent();

    // Set membership request permission on the teacher role.
    $group_role = $this->container->get('entity_type.manager')->getStorage('group_role')->load('lms_course-teacher');
    $group_role->grantPermission('administer membership requests');
    $group_role->save();
  }

  /**
   * Test membership request functionality.
   */
  public function testMembershipRequests(): void {
    $page = $this->getSession()->getPage();
    $assert_session = $this->assertSession();

    // Test additional course options on the second test course.
    $test_course_data = $this->courseData[2];
    $course = $this->getEntityByProperties('group', ['label' => $test_course_data['values']['label']]);
    $course->set('membership_validation', TRUE);
    $course->save();

    $this->drupalLogin($this->users['student']);
    $this->drupalGet(Url::fromRoute('entity.group.canonical', [
      'group' => $course->id(),
    ]));
    $page->clickLink('Request subscription');
    $page->pressButton('edit-submit');
    $assert_session->pageTextContains('Approval Pending');

    // Accept membership request as the teacher user.
    $this->drupalLogin($this->users['teacher']);

    $this->drupalGet(Url::fromRoute('entity.group.canonical', [
      'group' => $course->id(),
    ]));
    $this->screenShot();

    $membership_request_id = $this->getEntityIdByProperties('group_relationship', [
      'plugin_id' => 'group_membership_request',
      'gid' => $course->id(),
      'entity_id' => $this->users['student']->id(),
    ]);

    $this->drupalGet(Url::fromRoute('entity.group_relationship.group_approve_membership', [
      'group' => $course->id(),
      'group_relationship' => $membership_request_id,
    ]));
    $this->screenShot();

    $page->pressButton('edit-submit');
    $assert_session->pageTextContains('Membership request approved');

    $this->drupalLogin($this->users['student']);
    $this->drupalGet(Url::fromRoute('entity.group.canonical', [
      'group' => $course->id(),
    ]));
    $this->assertTrue($page->findLink('Start') !== NULL, 'Start link cannot be found on the page.');

    $this->watchdogTest();
  }

}
