<?php

declare(strict_types=1);

namespace Drupal\lms_migrate\Plugin\migrate\process;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\group\Entity\GroupInterface;
use Drupal\migrate\Attribute\MigrateProcess;
use Drupal\migrate\MigrateExecutableInterface;
use Drupal\migrate\Plugin\MigrationInterface;
use Drupal\migrate\ProcessPluginBase;
use Drupal\migrate\Row;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Get group relationships by plugin type.
 */
#[MigrateProcess('get_group_relationships')]
final class GetGroupRelationships extends ProcessPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The migration to be executed.
   *
   * @var \Drupal\migrate\Plugin\MigrationInterface
   */
  protected $migration;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a MigrationLookup object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\migrate\Plugin\MigrationInterface $migration
   *   The Migration the plugin is being used in.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, MigrationInterface $migration, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->migration = $migration;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition, ?MigrationInterface $migration = NULL) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $migration,
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\migrate\MigrateException
   */
  public function transform($value, MigrateExecutableInterface $migrate_executable, Row $row, $destination_property) {
    if (!is_array($value) || count($value) === 0) {
      return NULL;
    }

    [$gid, $plugin_id, $entity_id] = array_values($value);

    $group_storage = $this->entityTypeManager->getStorage('group');
    $group = $group_storage->load($gid);

    if (!($group instanceof GroupInterface)) {
      return NULL;
    }

    $relationships = $group->getRelationships($plugin_id);

    $filter_by_entity_id = $this->configuration['filter_by_entity_id'] ?? FALSE;
    if ($filter_by_entity_id && !is_null($entity_id)) {
      $relationships = array_filter($relationships, function ($relationship) use ($entity_id) {
        return $relationship->getEntity()->id() === $entity_id;
      });
    }

    $id_only = $this->configuration['id_only'] ?? FALSE;
    if ($id_only) {
      $relationships = array_map(function ($relationship) {
        return $relationship->id();
      }, $relationships);
    }

    return count($relationships) > 0 ? $relationships : NULL;
  }

}
