<?php

declare(strict_types=1);

namespace Drupal\lms_migrate\Plugin\migrate\process;

use Drupal\migrate\Attribute\MigrateProcess;
use Drupal\migrate\MigrateExecutableInterface;
use Drupal\migrate\ProcessPluginBase;
use Drupal\migrate\Row;

/**
 * Provides a lms_migrate_transformations plugin.
 */
#[MigrateProcess(
  id: "lms_migrate_transformations",
  handle_multiples: TRUE,
)]
final class Transformations extends ProcessPluginBase {

  /**
   * Filter out empty deltas and reset indexes.
   */
  public function singleArrayElementOrNull(mixed $value, MigrateExecutableInterface $migrate_executable, Row $row, string $destination_property): mixed {
    if (!\is_array($value) || \count($value) === 0) {
      return NULL;
    }
    elseif (\count($value) > 1) {
      $message = $this->configuration['message_multiple'] ?? "$destination_property: The provided source contains multiple values.";
      $migrate_executable->saveMessage($message);
      return NULL;
    }

    return reset($value);
  }

  /**
   * Filter out empty deltas and reset indexes.
   */
  public function filterFieldArray(mixed $value, MigrateExecutableInterface $migrate_executable, Row $row, string $destination_property): mixed {
    if (!\is_array($value)) {
      $migrate_executable->saveMessage("$destination_property: The provided source is not an array.");
      return NULL;
    }

    return \array_values(\array_filter($value));
  }

  /**
   * Lookup value in field array by key.
   */
  public function lookupValueInFieldArray(mixed $value, MigrateExecutableInterface $migrate_executable, Row $row, string $destination_property): mixed {
    [$needle, $haystack] = $value;

    if (!\is_array($haystack)) {
      $migrate_executable->saveMessage("$destination_property: The provided source does not include an array in the second position.");
      return NULL;
    }

    if (\count($haystack) === 0) {
      return NULL;
    }

    $lookup_key = $this->configuration['lookup_key'] ?? NULL;
    if (!$lookup_key || !\array_key_exists($lookup_key, $haystack[0])) {
      $migrate_executable->saveMessage("$destination_property: Lookup key not found field array.");
      return NULL;
    }

    $index = \array_search($needle, \array_column($haystack, $lookup_key), TRUE);
    if ($index === FALSE) {
      $migrate_executable->saveMessage("$destination_property: Value not found in field array.");
      return NULL;
    }

    return $index;
  }

}
