<?php

declare(strict_types=1);

namespace Drupal\lms_migrate\Plugin\migrate\source;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\migrate\Plugin\MigrationInterface;
use Drupal\migrate\Row;
use Drupal\migrate_drupal\Plugin\migrate\source\ContentEntity;
use Drupal\opigno_learning_path\Entity\Bundle\LearningPath;
use Drupal\opigno_learning_path\TrainingManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Group content entities.
 *
 * @MigrateSource(
 *   id = "content_entity:group",
 *   source_module = "migrate_drupal"
 * )
 */
final class ContentEntityGroup extends ContentEntity {

  /**
   * The training manager.
   */
  protected TrainingManager $trainingManager;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, MigrationInterface $migration, EntityTypeManagerInterface $entity_type_manager, EntityFieldManagerInterface $entity_field_manager, EntityTypeBundleInfoInterface $entity_type_bundle_info, TrainingManager $training_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $migration, $entity_type_manager, $entity_field_manager, $entity_type_bundle_info);
    $this->trainingManager = $training_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition, ?MigrationInterface $migration = NULL) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $migration,
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('opigno_learning_path.training_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function prepareRow(Row $row) {
    $result = parent::prepareRow($row);

    if ($row->hasSourceProperty('_learning_path')) {
      try {
        $modules_data = [];
        $learning_path = $row->getSourceProperty('_learning_path');

        // @todo Compare with result of opigno_learning_path.modules.get_items route.
        // @see \Drupal\opigno_learning_path\Controller\LearningPathContentController::getModules
        // @see \Drupal\opigno_learning_path\Controller\LearningPathContentController::sortModulesArray
        foreach ($this->trainingManager->getOrderedSteps($learning_path) as $module_id => $step) {
          $modules_data[$module_id] = [
            'id' => (int) $module_id,
            'required_score' => (int) $step['required_score'],
          ];

          // Might need to patch TrainingManager::getOrderedSteps method.
          if (array_key_exists('is_mandatory', $step)) {
            $modules_data[$module_id]['is_mandatory'] = (bool) $step['is_mandatory'];
          }
        }

        $row->setSourceProperty('_modules_data', array_values($modules_data));
        $row->setSourceProperty('_learning_path', NULL);
      }
      catch (\Exception $e) {
        $this->idMap->saveMessage($row->getSourceIdValues(), \trim($e->getMessage()), MigrationInterface::MESSAGE_WARNING);
      }
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  protected function toArray(ContentEntityInterface $entity): array {
    $return = parent::toArray($entity);

    if ($entity instanceof LearningPath) {
      $return['_learning_path'] = $entity;
    }

    return $return;
  }

}
