<?php

declare(strict_types=1);

namespace Drupal\lms_migrate\EventSubscriber;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Site\Settings;
use Drupal\migrate\Event\MigrateEvents;
use Drupal\migrate\Event\MigratePostRowSaveEvent;
use Drupal\path_alias\PathAliasInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Migrate event handler for lms_migrate module.
 */
final class MigrateEventSubscriber implements EventSubscriberInterface {

  /**
   * Constructs a MigrationEventSubscriber object.
   */
  public function __construct(
    protected readonly Settings $settings,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * Kernel response event handler.
   */
  public function onPostRowSave(MigratePostRowSaveEvent $event): void {
    $migration = $event->getMigration();
    $row = $event->getRow();

    if ($migration->id() === 'group__learning_path') {
      $course_path_prefix = Settings::get('lms_migrate.path_alias_rewrite_prefix.course');
      if (!is_string($course_path_prefix) || str_starts_with($course_path_prefix, '/') || !$row->hasSourceProperty('path')) {
        return;
      }

      $path = $row->getSourceProperty('path');
      if (!is_array($path) || count($path) === 0) {
        return;
      }

      [$alias, $pid, $langcode] = array_values($path[0]);
      $source_gid = $row->getSourceIdValues()['id'];
      $destination_gid = $event->getDestinationIdValues()[0];

      $storage = $this->entityTypeManager->getStorage('path_alias');
      $original = $storage->load($pid);
      assert($original instanceof PathAliasInterface);
      $original_path = $original->getPath();

      // Handle alias for source Opigno module. The original path_alias entity
      // is not updated to prevent a redirect from being created when the
      // redirect module is present and configure to auto create redirects.
      // @see \redirect_path_alias_update()
      $storage->delete([$original]);
      $storage->create([
        'path' => $original_path,
        'alias' => $course_path_prefix . $alias,
        'langcode' => $langcode,
      ])->save();

      // Create alias for newly migrated LMS course.
      $storage->create([
        'path' => str_replace($source_gid, $destination_gid, $original_path),
        'alias' => $alias,
        'langcode' => $langcode,
      ])->save();
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      MigrateEvents::POST_ROW_SAVE => ['onPostRowSave'],
    ];
  }

}
