<?php

declare(strict_types=1);

namespace Drupal\lms_webform\Plugin\ActivityAnswer;

use Drupal\Core\Entity\EntityFormBuilderInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\lms\Attribute\ActivityAnswer;
use Drupal\lms\Entity\Answer;
use Drupal\lms\Plugin\ActivityAnswerBase;
use Drupal\webform\WebformSubmissionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * File upload ActivityAnswer plugin.
 */
#[ActivityAnswer(
  id: 'webform',
  name: new TranslatableMarkup('Webform'),
)]
final class Webform extends ActivityAnswerBase implements PluginFormInterface {

  /**
   * The entity form builder.
   */
  protected EntityFormBuilderInterface $entityFormBuilder;

  /**
   * The created / edited webform submission.
   */
  protected ?WebformSubmissionInterface $webformSubmission = NULL;

  /**
   * {@inheritdoc}
   */
  public function evaluatedOnSave(Answer $answer): bool {
    return $this->configuration['evaluated_on_save'];
  }

  /**
   * {@inheritdoc}
   */
  public function injectServices(ContainerInterface $container): void {
    parent::injectServices($container);
    $this->entityFormBuilder = $container->get(EntityFormBuilderInterface::class);
  }

  /**
   * {@inheritdoc}
   *
   * @see \Drupal\file\Plugin\Field\FieldType\FileItem::fieldSettingsForm
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['evaluated_on_save'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Evaluated on submit'),
      '#description' => $this->t("Automatically set answers to activities of this type to evaluated and grant maximum score."),
      '#default_value' => $this->configuration['evaluated_on_save'] ?? FALSE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function answeringForm(array &$form, FormStateInterface $form_state, Answer $answer): void {
    $this->webformSubmission = $this->getWebformSubmission($answer);
    if ($this->webformSubmission === NULL) {
      $form['error'] = [
        '#theme' => 'status_messages',
        '#message_list' => [
          'error' => [$this->t('Webform no longer exists.')],
        ],
        '#status_headings' => [
          'error' => $this->t('Error'),
        ],
      ];
      return;
    }

    $form['webform_submission'] = [
      '#type' => 'container',
    ];

    $subform_object = $this->entityTypeManager->getFormObject('webform_submission', $this->webformSubmission->isNew() ? 'add' : 'edit');
    $subform_object->setEntity($this->webformSubmission);
    $subform_state = SubformState::createForSubform($form['webform_submission'], $form, $form_state, $subform_object);
    $form['webform_submission'] = $subform_object->buildForm($form['webform_submission'], $subform_state);
    unset($form['webform_submission']['information']);
    unset($form['webform_submission']['actions']);
  }

  /**
   * {@inheritdoc}
   */
  public function validateAnsweringForm(array &$form, FormStateInterface $form_state, Answer $answer): void {
    parent::validateAnsweringForm($form, $form_state, $answer);
    if ($this->webformSubmission === NULL) {
      return;
    }

    $subform_object = $this->entityTypeManager->getFormObject('webform_submission', $this->webformSubmission->isNew() ? 'add' : 'edit');
    $subform_object->setEntity($this->webformSubmission);
    $subform_state = SubformState::createForSubform($form['webform_submission'], $form, $form_state, $subform_object);
    $this->webformSubmission = $subform_object->validateForm($form['webform_submission'], $subform_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitAnsweringForm(array &$form, FormStateInterface $form_state, Answer $answer): void {
    if ($this->webformSubmission === NULL) {
      return;
    }

    $subform_object = $this->entityTypeManager->getFormObject('webform_submission', $this->webformSubmission->isNew() ? 'add' : 'edit');
    $subform_object->setEntity($this->webformSubmission);
    $subform_state = SubformState::createForSubform($form['webform_submission'], $form, $form_state, $subform_object);
    $subform_object->submitForm($form['webform_submission'], $subform_state);
    $subform_object->save($form['webform_submission'], $subform_state);

    $answer->setData(['submission_id' => $subform_object->getEntity()->id()]);
  }

  /**
   * {@inheritdoc}
   */
  protected function getAnswerRenderable(Answer $answer): array {
    $submission = $this->getWebformSubmission($answer);
    if ($submission === NULL) {
      return [
        '#markup' => $this->t('No submission found.'),
      ];
    }
    return $this->entityTypeManager->getViewBuilder('webform_submission')->view($submission);
  }

  /**
   * Get webform submission for this activity's answer.
   */
  private function getWebformSubmission(Answer $answer): ?WebformSubmissionInterface {
    $webform_field = $answer->getActivity()->get('webform');
    $webform = NULL;
    if (!$webform_field->isEmpty()) {
      /** @var \Drupal\webform\WebformInterface */
      $webform = $webform_field->entity;
    }
    if ($webform === NULL) {
      return NULL;
    }

    $webform_submission = NULL;
    $data = $answer->getData();
    if (\array_key_exists('submission_id', $data)) {
      $webform_submission = $this->entityTypeManager->getStorage('webform_submission')->load($data['submission_id']);
    }
    if ($webform_submission === NULL) {
      $webform_submission = $this->entityTypeManager->getStorage('webform_submission')->create([
        'webform_id' => $webform->id(),
        'entity_type' => 'lms_activity',
        'entity_id' => $answer->getActivity()->id(),
      ]);
    }
    return $webform_submission;
  }

}
