<?php

declare(strict_types=1);

namespace Drupal\Tests\lms_file_upload\FunctionalJavascript;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Url;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\lms\Entity\AnswerInterface;
use Drupal\Tests\lms\LmsTestHelperTrait;

/**
 * General javascript test of LMS basic features.
 *
 * @group lms
 */
final class LmsWebformActivityTest extends WebDriverTestBase {

  use LmsTestHelperTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'dblog',
    'block',
    'page_cache',
    'dynamic_page_cache',
    'lms',
    'lms_classes',
    'lms_webform',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'claro';

  /**
   * Test user accounts.
   */
  private array $users = [];

  /**
   * Test data to pass between methods.
   */
  private array $testData = [];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Import test config.
    $this->importTestConfig();

    // Create test users.
    $users_source_file_path = $this->container->get(ModuleExtensionList::class)->getPath('lms') . '/tests/data/users.yml';
    $user_data = Yaml::decode(\file_get_contents($users_source_file_path));
    foreach ($user_data as $name => $values) {
      $this->users[$name] = $this->drupalCreateUser([], NULL, FALSE, $values);
    }

    // Create test webform.
    $this->testData['webform'] = Yaml::decode(\file_get_contents(__DIR__ . '/../../source_data/test_webform.yml'));
    $webform = $this->container->get('entity_type.manager')->getStorage('webform')
      ->create($this->testData['webform']);
    $webform->save();

    $this->testData['webform']['answers'] = [];
  }

  /**
   * Test router.
   */
  public function testLmsFileUpload(): void {
    $this->activityTypeTest();
    $this->courseCreationTest();
    $this->activityTakingTest();
    $this->courseResultsTest();
    $this->watchdogTest();
  }

  /**
   * Create LMS file upload activity type.
   */
  private function activityTypeTest(): void {
    $page = $this->getSession()->getPage();
    $assert_session = $this->assertSession();

    $this->drupalLogin($this->users['admin']);

    // Activity type creation test.
    $this->drupalGet(Url::fromRoute('entity.lms_activity_type.collection'));

    $page->clickLink('Add activity type');
    $this->setFormElementValue('input', 'name', 'Webform');
    $this->setFormElementValue('select', 'pluginId', 'webform');

    $assert_session->waitForElementVisible('css', '[name^="pluginConfiguration"]');
    $page->checkField('pluginConfiguration[evaluated_on_save]');
    $this->pressButton('edit-submit');

    self::assertTrue(\strpos(\strip_tags($page->getHtml()), 'Webform') !== FALSE, 'The Webform activity type is not installed.');
  }

  /**
   * Course creation test.
   */
  private function courseCreationTest(): void {
    $this->drupalLogin($this->users['teacher']);

    // Create test activity.
    $activity_name = 'Test webform activity';
    $this->drupalGet(Url::fromRoute('entity.lms_activity.add_form', [
      'lms_activity_type' => 'webform',
    ]));
    $this->setEntityFormField('name', $activity_name);
    $this->getSession()->getPage()->selectFieldOption('webform[0][target_id]', 'test_webform');
    $this->pressButton('edit-submit');
    $this->testData['activity'] = $this->getEntityByProperties('lms_activity', [
      'name' => $activity_name,
    ]);

    // Create a lesson.
    $lesson_name = 'Test webform lesson';
    $this->drupalGet(Url::fromRoute('entity.lms_lesson.add_form', [
      'lms_lesson_type' => 'lesson',
    ]));
    $this->setEntityFormField('name', $lesson_name);
    $this->setLmsReferenceField('activities', [['entity' => $this->testData['activity']]]);
    $this->pressButton('edit-submit');
    $this->testData['lesson'] = $this->getEntityByProperties('lms_lesson', [
      'name' => $lesson_name,
    ]);

    // Create a course.
    $course_name = 'Test webform course';
    $this->drupalGet(Url::fromRoute('entity.group.add_form', [
      'group_type' => 'lms_course',
    ]));
    $this->setEntityFormField('label', $course_name);
    $this->setLmsReferenceField('lessons', [['entity' => $this->testData['lesson']]]);
    $this->pressButton('edit-submit');
    $this->testData['course'] = $this->getEntityByProperties('group', [
      'label' => $course_name,
    ]);
  }

  /**
   * Activity take test.
   */
  private function activityTakingTest(): void {
    $page = $this->getSession()->getPage();

    $this->drupalLogin($this->users['student']);

    $this->drupalGet(Url::fromRoute('entity.group.canonical', [
      'group' => $this->testData['course']->id(),
    ]));
    $page->clickLink('Enroll');
    $this->pressButton('edit-submit');
    $page->clickLink('Start');

    $this->testData['webform']['answers']['some_checkbox'] = (bool) \rand(0, 1);
    if ($this->testData['webform']['answers']['some_checkbox']) {
      $page->checkField('some_checkbox');
    }
    $this->testData['webform']['answers']['tell_us_more'] = $this->randomString(20);
    $page->fillField('tell_us_more', $this->testData['webform']['answers']['tell_us_more']);
    $this->pressButton('[data-drupal-selector="edit-submit"]', 'css');

    // Test if saved submission values are saved and visible when revisiting.
    $page->clickLink('Revisit');
    if ($this->testData['webform']['answers']['some_checkbox']) {
      self::assertTrue($page->hasCheckedField('some_checkbox'));
    }
    else {
      self::assertTrue($page->hasUncheckedField('some_checkbox'));
    }

    self::assertSame($this->testData['webform']['answers']['tell_us_more'], $page->find('css', '#edit-tell-us-more')->getValue());
  }

  /**
   * Course results test.
   */
  private function courseResultsTest(): void {
    $page = $this->getSession()->getPage();

    $this->drupalLogin($this->users['teacher']);
    $student_id = $this->users['student']->id();

    // Results page.
    $this->drupalGet(Url::fromRoute('lms.group.results', [
      'group' => $this->testData['course']->id(),
      'user' => $student_id,
    ]));

    $page->find('css', '.lesson-score-details')->click();

    $answer_id = $this->getEntityIdByProperties('lms_answer', [
      // Skip lesson status condition to save some code - as long as
      // this activity is used only in a single lesson, all is fine.
      'user_id' => $student_id,
      AnswerInterface::ACTIVITY_FIELD => $this->testData['activity']->id(),
    ]);

    $details_uri = Url::fromRoute('lms.answer.details', [
      'lms_answer' => $answer_id,
      'js' => 'nojs',
    ])->toString();
    $this->getSession()->getPage()->find('css', '[href="' . $details_uri . '"]')->click();
    $modal = $this->assertSession()->waitForElementVisible('css', '[role="dialog"]');

    $test_webform_elements = Yaml::decode($this->testData['webform']['elements']);
    foreach ($this->testData['webform']['answers'] as $element_id => $answer) {
      $element = $test_webform_elements[$element_id];
      if ($element['#type'] === 'checkbox') {
        $expected_answer = $answer ? 'Yes' : 'No';
      }
      else {
        $expected_answer = $answer;
      }
      $wrapper = $modal->find('css', '#' . $this->testData['webform']['id'] . '--' . $element_id);
      self::assertEquals($element['#title'], $wrapper->find('css', '.form-item__label')->getText());
      $this->assertElementTextContains($wrapper, $expected_answer, 'Expected answer not found.');
    }
  }

}
