<?php

namespace Drupal\localgov_entity_share\Plugin\EntityShareClient\Processor;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\entity_share\EntityShareUtility;
use Drupal\entity_share_client\Event\RelationshipFieldValueEvent;
use Drupal\entity_share_client\ImportProcessor\ImportProcessorPluginBase;
use Drupal\entity_share_client\RuntimeImportContext;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Filters out LocalGov parent reference fields from relationships.
 *
 * This prevents them from being followed by the Entity Reference processor
 * and prevents circularity, as these fields have a corresponding child
 * reference field pointing in the opposite direction.
 *
 * The value of the reference field is not affected.
 *
 * This is weighted to run before the entity_reference processor, and removes
 * the relationship data before that handles the entity.
 *
 * @ImportProcessor(
 *   id = "localgov_reference_field_filter",
 *   label = @Translation("Localgov parent reference field filter"),
 *   description = @Translation("Prevents LocalGov-related parent reference fields from being followed by the Entity Reference processor. Must run before the Entity Reference processor."),
 *   stages = {
 *     "process_entity" = -20,
 *   },
 * )
 */
class LocalgovReferenceFieldFilter extends ImportProcessorPluginBase {

  /**
   * The LocalGov reference fields to filter out.
   */
  const LOCALGOV_FILTERED_FIELDS = [
    'node' => [
      'localgov_services_parent',
      'localgov_step_parent',
      'localgov_guides_parent',
      'localgov_subsites_parent',
    ],
  ];

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The event dispatcher service.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  protected $eventDispatcher;

  /**
   * The logger channel service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $loggerChannel;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('event_dispatcher'),
      $container->get('logger.channel.entity_share_client'),
    );
  }

  /**
   * Creates a LocalgovReferenceFieldFilter instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   The event dispatcher service.
   * @param \Psr\Log\LoggerInterface $logger_channel
   *   The logger channel service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityTypeManagerInterface $entity_type_manager,
    EventDispatcherInterface $event_dispatcher,
    LoggerInterface $logger_channel,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->eventDispatcher = $event_dispatcher;
    $this->loggerChannel = $logger_channel;
  }

  /**
   * {@inheritdoc}
   */
  public function processEntity(RuntimeImportContext $runtime_import_context, ContentEntityInterface $processed_entity, array $entity_json_data) {
    $field_mappings = $runtime_import_context->getFieldMappings();
    $entity_type_id = $processed_entity->getEntityTypeId();
    $entity_bundle = $processed_entity->bundle();

    // Skip this entity if we have nothing to say about the entity type.
    if (!isset(static::LOCALGOV_FILTERED_FIELDS[$entity_type_id])) {
      return;
    }

    $save = FALSE;
    foreach (static::LOCALGOV_FILTERED_FIELDS[$entity_type_id] as $field_name) {
      // Skip this field if it's not present on the current entity's bundle.
      if (!isset($field_mappings[$entity_type_id][$entity_bundle][$field_name])) {
        continue;
      }

      $field_public_name = $field_mappings[$entity_type_id][$entity_bundle][$field_name];

      if (empty($entity_json_data['relationships'][$field_public_name]['data'])) {
        continue;
      }


      $field = $processed_entity->get($field_name);
      $main_property = $field->getItemDefinition()->getMainPropertyName();

      $field_data = $entity_json_data['relationships'][$field_public_name];
      $prepared_field_data = EntityShareUtility::prepareData($field_data['data']);

      $referenced_entities_ids = $this->getExistingEntities($prepared_field_data);

      // Add field value.
      // As the loop is on the JSON:API data, the sort is preserved.
      foreach ($prepared_field_data as $field_value_data) {
        $referenced_entity_uuid = $field_value_data['id'];

        // Check that the referenced entity exists or had been imported.
        if (!isset($referenced_entities_ids[$referenced_entity_uuid])) {
          continue;
        }

        $field_value = [
          $main_property => $referenced_entities_ids[$referenced_entity_uuid],
        ];
        // Add field metadata.
        if (isset($field_value_data['meta'])) {
          $field_value += $field_value_data['meta'];
        }

        // Allow to alter the field value with an event.
        $event = new RelationshipFieldValueEvent($field, $field_value);
        $this->eventDispatcher->dispatch($event, RelationshipFieldValueEvent::EVENT_NAME);
        $field_values[] = $event->getFieldValue();
      }
      $processed_entity->set($field_name, $field_values);
      $save = TRUE;
    }

    if ($save) {
      // Save the entity once all the references have been updated.
      $processed_entity->save();
    }
  }

  /**
   * Helper function to get existing reference entities.
   *
   * Identical to
   * Drupal\entity_share_client\Plugin\EntityShareClient\Processor\EntityReference::getExistingEntities().
   *
   * @param array $data The JSON:API data for an entity reference field.
   *
   * @return array An array of entity IDs keyed by UUID.
   */
  protected function getExistingEntities(array $data) {
    $referenced_entities_ids = [];
    $entity_uuids = [];

    // Extract list of UUIDs.
    foreach ($data as $field_value_data) {
      if ($field_value_data['id'] !== 'missing') {
        $parsed_type = explode('--', $field_value_data['type']);
        $entity_type_id = $parsed_type[0];
        $entity_uuids[] = $field_value_data['id'];
      }
    }

    if (!empty($entity_uuids)) {
      try {
        // Load the entities to be able to return an array of IDs keyed by
        // UUIDs. Sorting the array will be done later.
        $entity_storage = $this->entityTypeManager->getStorage($entity_type_id);
        $existing_entity_ids = $entity_storage->getQuery()
          ->accessCheck(FALSE)
          ->condition('uuid', $entity_uuids, 'IN')
          ->execute();

        $existing_entities = $entity_storage->loadMultiple($existing_entity_ids);
        foreach ($existing_entities as $existing_entity) {
          $referenced_entities_ids[$existing_entity->uuid()] = $existing_entity->id();
        }
      }
      catch (\Exception $e) {
        $log_variables = [];
        $log_variables['@msg'] = $e->getMessage();
        $this->logger->error('Caught exception trying to load existing entities. Error message was @msg', $log_variables);
      }
    }

    return $referenced_entities_ids;
  }

}
