This module contains an example Webform configuration demonstrating integration with a REST API created on the Netcall Liberty Create Low-code platform.

## Liberty Create API
Liberty Create users can create custom APIs from a drag-and-drop UI.  Due to the custom nature of these APIs, we cannot provide a single integration for Drupal Webforms.  This means when faced with an integration task, Drupal site builders would have to study every such API method and come up with suitable Webform configurations.

Caselink 360 is a CRM app created on the Liberty Create platform.  It comes with a published API reference.  This makes it possible for us to provide an example Webform integration with one of Caselink 360's REST API methods.  This API method creates CRM case records.

The rest of this README explains the Webform configuration process with the hope that this explanation would be useful when integrating Drupal Webforms with other REST APIs built on the Liberty Create platform.

The example Webform provided in this module creates new CRM case records using the Caselink 360 app's `caselink_360_create_update_case` REST API method.  First we take a look at a sample API request, then a sample API response, and finally explain the Webform configuration process.

## Example API request
Below we look at a REST API request to the `caselink_360_create_update_case` API method that creates a CRM case record:

```
POST https://example-build.oncreate.app/api/REST/case_to_crm/0.1 HTTP/1.1
API-Authentication: [Token hidden]
Content-Type: application/json

{
    "payload":{
        "client_unique_identifier":"63105fe17248615689a02568ca95ff91",
        "function":"caselink_360_create_update_case",
        "data":[
            {
                "source_system":"Some Text 1",
                "source_ref":"Some Text 2",
                "nature_of_enquiry":"Some Text 5",
                "case_datetime_created":"25/12/2023 12:00",
                "resident_uprn":75,
                "case_uprn":75,
                "case_url":"http://www.example.com",
                "disposal_date":"25/12/2023",
                "details":"Some Long Text 1",
                "first_name":"Some Text 3",
                "last_name":"Some Text 4",
                "date_of_birth":"25/12/2003",
                "email_address":"nobody@example.com",
                "phone_number":"07700900000",
                "documents":[
                    {
                        "file":{
                            "filename":"Example.txt",
                            "is_base64":true,
                            "content":"U29tZSBUZXh0IDE="
                        },
                        "filename":"Some Text 6",
                        "description":"Some Text 7"
                    }
                ]
            }
        ]
    }
}
```

## Example API response
What follows is an example response from the previous API call to create a CRM case record:

```
{
    "payload":{
        "client_unique_identifier":"63105fe17248615689a02568ca95ff91",
        "result":"success",
        "error_code":null,
        "error_desc":null,
        "warnings":[
        ],
        "data":[
            {
                "result":"created",
                "error_code":null,
                "error_desc":null,
                "data":{
                    "source_system":"Some Text 1",
                    "source_ref":"Some Text 2",
                    "case_datetime_created":"1703505600",
                    "ch_case_ref":"GE\/1"
                }
            }
        ]
    }
}
```

## Webform integration basics
This is how we setup API integration for the `caselink_360_create_update_case` API method:
1. Under the "Settings > Emails/Handlers" tab of a Webform, add a "Remote post" Webform handler.  This handler comes bundled with the Webform module.  Note down the handler id which is usually autogenerated but can be edited manually if wished.
2. Under the "General" tab of the handler configuration dialog, expand the "Completed" fieldset.
3. In the "Completed URL" field, enter the API endpoint URL.  The example Webform config bundled with this module uses "https://example-build.oncreate.app/api/REST/case_to_crm/0.1".  Adjust it accordingly for your target endpoint.
4. In the "Completed custom data" field, map API fields with suitable Webform tokens.  Refer to the example config below to get a better idea.  This bit is in YAML format:
  ```
  payload:
      # client_unique_identifier is a required field.  Its value should vary between requests.
      client_unique_identifier: "[webform:id]/[webform_submission:sid]"
      # "function" is also a required field.
      function: caselink_360_create_update_case
      # So is "data".
      data:
        -
          # source_system is a required field.
          source_system: "Drupal Webforms" 
          # source_ref is a required field.
          source_ref: "[webform:id]/[webform_submission:sid]"
          # nature_of_enquiry is a required field.
          nature_of_enquiry: "[webform:title]"
          # Everything else below is optional.
          case_datetime_created: "[webform_submission:completed:custom:d/m/Y H:i]"
          resident_uprn: "[webform_submission:values:residential_address:uprn]"
          case_uprn: "[webform_submission:values:case_address:uprn]"
          first_name: "[webform_submission:values:name:first]"
          last_name: "[webform_submission:values:name:last]"
          phone_number: "[webform_submission:values:phone]"
          email_address: "[webform_submission:values:email]"
          date_of_birth: "[webform_submission:values:date_of_birth:d/m/Y]"
          case_url:  "[webform_submission:token-view-url]"
          disposal_date: "[webform_submission:purge_date:custom:d/m/Y:clear]"
          #
          # Any other Webform submission token should be placed within the "details" field below.
          details: |-
            Case address: "[webform_submission:values:case_address:clear]"
            Residential address: "[webform_submission:values:residential_address:clear]"
            Details: "[webform_submission:values:details_of_enquiry:clear]"
          # For file fields, we use inline YAML syntax to avoid indentation issues.  This is because file_details_for_liberty_create_api, our custom file token, gets replaced with several *other* tokens before the token value insertion round starts.
          documents: ["[webform_submission:values:files:file_details_for_liberty_create_api]", "[webform_submission:values:more_files:file_details_for_liberty_create_api]"]
  ```

  This assumes our Webform carries at least the following fields:
  - name
  - date_of_birth
  - email
  - phone
  - residential_address
  - case_address
  - files
  - more_files
5. Uncheck everything under the "Submission data" fieldset.
6. Switch to the "Advanced" tab of the "Remote post" handler.
7. Under "Additional settings", select "POST" from the "Method" dropdown.
8. Select "JSON" from the "Post type" dropdown.
9. Insert the following snippet in "Custom options":
  ```
  headers:
    API-Authentication: "[env:DRUPAL_LIBERTY_CREATE_API_AUTH_KEY]"
  ```
  This assumes that the [token_environment Drupal module](https://www.drupal.org/project/token_environment) is enabled and the DRUPAL_LIBERTY_CREATE_API_AUTH_KEY environment variable is present with its corresponding value.

  Note that the token_environment module must be explicitly told that the DRUPAL_LIBERTY_CREATE_API_AUTH_KEY environment variable should be made available as a Drupal token.  This is configured from "/admin/config/system/token-environment".

10. The [webform_queued_post_handler Drupal module](https://packagist.org/packages/cyberwoven/webform_queued_post_handler) provides an alternative to the "Remote post" handler.  This handler is called "Async remote post" and uses Drupal's queue to manage API requests which is more reliable.  Items in Drupal's queue are usually processed during cron runs.  If you decide to use this handler instead, you may also find the [queue_ui](https://www.drupal.org/project/queue_ui) contrib module useful.

### Note
If you install this module, almost all the above steps would be taken care of you except:
- You would have to provide the right API endpoint URL referred to in step 3 above.
- Make the API token value available as the DRUPAL_LIBERTY_CREATE_API_AUTH_KEY environment variable as mentioned in step 9.

It is unlikely you would want to use the Caselink 360 integration Webform as is.  Treat it more as an example to build up on rather than an end product.

## Inspecting API responses
To inspect API responses, add the following "Value" type Webform elements to your Webform:
- "crm_response" whose value should be `[webform:handler:remote_post:completed:payload:result]; [webform:handler:remote_post:completed:payload:error_code]; [webform:handler:remote_post:completed:payload:error_desc]`.
- "crm_result" whose value should be `[webform:handler:remote_post:completed:payload:data:0:result]; [webform:handler:remote_post:completed:payload:data:0:error_code]; [webform:handler:remote_post:completed:payload:data:0:error_desc]`
- "crm_case_ref" whose value should be `[webform:handler:remote_post:completed:payload:data:0:data:0:ch_case_ref]`.

This will ensure that API responses are stored alongside Webform submission values.  This makes it easier to inspect these from the "Results" tab of Webforms.

As you can see, the three above field values are using multiple tokens and these tokens are referring to `remote_post` which is a handler id.  This can vary dependencing on what handler id is in use.  Everything after ":completed:" in the token mirrors the API response.  Change all these if necessary.

## Summary
- Liberty Create APIs will vary from organisation to organisation.  There is no one-size fits all solution.  You can study the example Webform config provided with this module to get an idea about the integration process.  This example creates CRM case records in the Caselink 360 Liberty Create app.
- Use the "Remote post" or "Async remote post" Webform handler to make HTTP POST requests to any Liberty Create REST API endpoint URL.
- Use the "Completed custom data" settings of the above handlers to map Webform fields to REST API fields.
- Use the "Custom options" settings of the handlers to provide API authentication details.
- Use the crm_response, crm_result, and crm_case_ref *Value* type Webform fields to capture API responses.
