<?php

namespace Drupal\localgov_publications_importer\Commands;

use Drupal\Core\Cache\Cache;
use Drupal\localgov_publications_importer\Service\ImportManager;
use Drush\Commands\DrushCommands;

/**
 * Drush commands for PDF import.
 */
class ImportCommand extends DrushCommands {

  /**
   * Constructor.
   */
  public function __construct(
    protected ImportManager $importManager,
  ) {
    parent::__construct();
  }

  /**
   * Process pending imports.
   *
   * @param int $limit
   *   Number of imports to process. This must be a positive integer. Default 1.
   *
   * @command localgov_publications_importer:import
   * @aliases lpii
   * @usage localgov_publications_importer:import 10
   *   Process ten pending imports.
   */
  public function import($limit = 1): void {

    $validLimit = FALSE;
    if (is_numeric($limit)) {
      $limit = (int) $limit;
      if ($limit > 0) {
        $validLimit = TRUE;
      }
    }

    if (!$validLimit) {
      $this->logger()->error('Limit param must be a positive integer.');
      return;
    }

    $processedCount = 0;
    for ($i = 0; $i < $limit; $i++) {
      if ($this->importManager->importNext()) {
        $processedCount++;
      }
      else {
        break;
      }
    }

    if ($processedCount > 0) {
      $this->logger()->success("Processed {$processedCount} import(s).");
    }
    else {
      $this->logger()->notice('No pending imports to process.');
    }

    // For some reason, saving the imports invalidates the cached view once when
    // an imports start, but not when an import ends. Invalidating this tag for
    // the imports view means that imports ending show up the view. This should
    // be considered a workaround though, as it shouldn't be required.
    Cache::invalidateTags(['config:views.view.imports']);
  }

  /**
   * Resets the status of a given import, so it can be reimported.
   *
   * @param int $import_id
   *   The id of the import to reset.
   *
   * @command localgov_publications_importer:reset
   * @aliases lpir
   */
  public function reset(int $import_id): void {
    $this->importManager->reset($import_id);
  }

}
