<?php

namespace Drupal\localgov_publications_importer\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\localgov_publications_importer\Entity\Import;
use Drupal\localgov_publications_importer\ImportInterface;

/**
 * Service for managing the import process.
 */
class ImportManager {

  use LoggerChannelTrait;

  /**
   * Constructor.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected Importer $importer,
  ) {
  }

  /**
   * Get the next import that needs processing and process it if one exists.
   *
   * @return bool
   *   TRUE if an import was processed, FALSE otherwise.
   */
  public function importNext(): bool {

    /** @var \Drupal\localgov_publications_importer\ImportStorage $importStorage */
    $importStorage = $this->entityTypeManager->getStorage('import');
    $import = $importStorage->getNextToImport();

    if (!$import instanceof ImportInterface) {
      return FALSE;
    }

    try {
      // Mark the import as processing before we begin.
      $import->setStatus(Import::STATUS_PROCESSING);
      $import->save();

      $node = $this->importer->doImport($import);

      if ($node === NULL) {
        throw new \Exception('Import failed.');
      }

      $this->getLogger('localgov_publications_importer')->info("Successfully imported import @import as node @node", [
        '@import' => $import->id(),
        '@node' => $node->id(),
      ]);
    }
    catch (\Throwable $e) {
      $this->getLogger('localgov_publications_importer')->error($e->getMessage(), ['exception' => $e]);
      $import->setStatus(Import::STATUS_FAILED);
      $import->save();
      return TRUE;
    }

    // If the process didn't fail, mark it as completed.
    $import->setStatus(Import::STATUS_COMPLETED);
    $import->save();
    return TRUE;
  }

  /**
   * Resets the status of a given import, so it can be reimported.
   */
  public function reset(int $import_id) {

    $importStorage = $this->entityTypeManager->getStorage('import');

    $import = $importStorage->load($import_id);
    if ($import instanceof ImportInterface) {
      $import->setStatus(Import::STATUS_PENDING);
      $import->setPages([]);
      $import->setImages([]);
      $import->save();
    }
  }

}
