<?php

declare(strict_types=1);

namespace Drupal\localgov_waste_collection;

use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Base class for DataProvider plugins.
 */
abstract class DataProviderBase extends PluginBase implements DataProviderInterface {

  /**
   * Fetches addresses matching the given postcode.
   *
   * @param string $postcode
   *   The postcode to search for.
   *
   * @return array
   *   An array of matching addresses.
   *
   * @throws \Drupal\localgov_waste_collection\Exception\PostcodeSearchException
   *   Throws an exception if there is a problem with the request to the
   *   data provider.
   */
  abstract public function findAddressesByPostcode(string $postcode): array;

  /**
   * Fetches the address for a UPRN.
   *
   * @param string $uprn
   *   The UPRN of the property.
   *
   * @return ?string
   *   The address of the string, or NULL if not found.
   */
  abstract public function getAddressFromUprn(string $uprn): ?string;

  /**
   * Fetches the waste collection dates for a UPRN.
   *
   * @param string $uprn
   *   A string containing the UPRN to match.
   *
   * @return array
   *   A sorted, associative array of collection dates, see README.md for
   *   details.
   */
  abstract public function getCollections(string $uprn): array;

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration(): array {
    return $this->configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration): void {
    $this->configuration = $configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {}

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {}

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies(): array {
    return [];
  }

}
