<?php

declare(strict_types=1);

namespace Drupal\localgov_waste_collection\Controller;

use Drupal\Component\Utility\Unicode;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\TempStore\PrivateTempStore;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Url;
use Drupal\localgov_waste_collection\DataProviderManager;
use Drupal\localgov_waste_collection\Exception\PostcodeSearchException;
use Drupal\localgov_waste_collection\Service\PublicHolidays;
use Spatie\IcalendarGenerator\Components\Calendar;
use Spatie\IcalendarGenerator\Components\Event;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;
use Symfony\Component\HttpFoundation\StreamedResponse;

/**
 * Waste collection schedule controller.
 *
 * @package Drupal\localgov_waste_collection\Controller
 */
class WasteCollectionController extends ControllerBase {

  /**
   * The data provider plugin manager.
   *
   * @var \Drupal\localgov_waste_collection\DataProviderManager
   */
  protected $dataProviderManager;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The data provider plugin.
   *
   * @var \Drupal\localgov_waste_collection\DataProviderBase
   */
  protected $dataProvider;

  /**
   * The form builder interface.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * The public holiday provider.
   *
   * @var \Drupal\localgov_waste_collection\Service\PublicHolidays
   */
  protected $publicHolidays;

  /**
   * Private temp storage.
   */
  private PrivateTempStore $tempStore;

  /**
   * The configuration object.
   *
   * @var \Drupal\Core\Config\Config
   */
  private Config $config;

  /**
   * Constructor.
   */
  public function __construct(PrivateTempStoreFactory $temp_store_factory, DataProviderManager $dataProviderManager, ConfigFactoryInterface $configFactory, FormBuilderInterface $formBuilder, PublicHolidays $publicHolidays) {
    $this->tempStore = $temp_store_factory->get('localgov_waste_collection_schedule');
    $this->dataProviderManager = $dataProviderManager;
    $this->configFactory = $configFactory;
    $this->formBuilder = $formBuilder;
    $this->publicHolidays = $publicHolidays;

    $this->config = $this->configFactory->get('localgov_waste_collection.settings');
    $plugin_id = $this->config->get('active_data_provider');
    $this->dataProvider = $this->dataProviderManager->createInstance($plugin_id);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('tempstore.private'),
      $container->get('plugin.manager.localgov_waste_collection.data_provider'),
      $container->get('config.factory'),
      $container->get('form_builder'),
      $container->get('localgov_waste_collection.public_holidays_provider'),
    );
  }

  /**
   * Display the postcode form.
   *
   * @return array
   *   Drupal render array
   */
  public function showForm(?Request $request): array {
    $form = $this->formBuilder->getForm('Drupal\localgov_waste_collection\Form\PostcodeForm');
    $build['#form'] = $form;

    $build['#theme'] = 'localgov_waste_collection_form';

    if ($this->tempStore->get('noresults_postcode')) {
      $build['#noresults'] = $this->tempStore->get('noresults_postcode');
      $this->tempStore->delete('noresults_postcode');
    }
    return $build;
  }

  /**
   * Display the results of a postcode search.
   *
   * @return array|RedirectResponse
   *   Drupal render array or redirect back to form.
   */
  public function getAddresses(Request $request): array|RedirectResponse {
    $query_parameters = $request->query->all();
    $postcode = urldecode($query_parameters["postcode"]);

    $properties = [];
    try {
      $properties = $this->dataProvider->findAddressesByPostcode($postcode);
    }
    catch (PostcodeSearchException $e) {
      // @todo This contradicts the 'No properties found for postcode...'
      // message which is output when there are not properties found!
      \Drupal::messenger()
        ->addError($this->t('There was a problem with the postcode search. Please try again later.'));
    }

    if (empty($properties)) {
      $this->tempStore->set('noresults_postcode', $postcode);
      return $this->redirect('localgov_waste_collection.form');
    }

    $form = $this->formBuilder->getForm('Drupal\localgov_waste_collection\Form\AddressSelectForm', $properties);

    $build[] = [
      '#form' => $form,
      '#theme' => 'localgov_waste_collection_property_list',
      '#postcode' => $postcode,
      '#properties' => $properties,
    ];

    return $build;
  }

  /**
   * View the collection schedule for a UPRN.
   *
   * @param string $uprn
   *   UPRN to lookup.
   *
   * @return array
   *   Drupal render array
   */
  public function viewSchedule(string $uprn): array {
    $build = [];
    $collections = $this->dataProvider->getCollections($uprn);

    if ($this->config->get('public_holidays')) {
      $publicHolidays = $this->publicHolidays->getPublicHolidays();
      foreach ($collections["dates"] as $key => $values) {
        $formattedDate = date('d-m-Y', strtotime($values["date"]));
        $holiday = $publicHolidays[$formattedDate] ?? NULL;
        $collections["dates"][$key]["holiday"] = $holiday;
      }
    }

    // Assemble the collection type labels of next collection day.
    $first_collection_date = NULL;
    $next_collection_labels = [];
    foreach ($collections['dates'] as $collection) {
      // Store the first date that's in the future.
      if (is_null($first_collection_date) && $collection['date'] > date('Y-m-d')) {
        $first_collection_date = $collection['date'];
      }

      if ($collection['date'] == $first_collection_date) {
        $next_collection_labels[] = $collection['type']['label'];
      }
    }
    if ($first_collection_date) {
      $build['#next_collection_date'] = strtotime($first_collection_date);
      $build['#next_collection_labels'] = $next_collection_labels;
    }

    // Assemble the calendar range label.
    if (!empty($collections['dates'])) {
      $first_collection_timestamp = strtotime(reset($collections['dates'])['date']);
      $last_collection_timestamp = strtotime(end($collections['dates'])['date']);
      $build['#calendar_range_label'] = $this->t('@start-month to @end-month', [
        '@start-month' => date(
          date('Y', $first_collection_timestamp) == date('Y', $last_collection_timestamp) ? 'F' : 'F Y',
          $first_collection_timestamp,
        ),
        '@end-month' => date('F Y', $last_collection_timestamp),
      ]);
    }

    $dates = [];
    $colour_theme_keys = [];
    foreach ($collections['dates'] as $index => $collection) {
      if (!isset($colour_theme_keys[$collection['type']['colour']])) {
        $colour_theme_keys[$collection['type']['colour']] = preg_replace('/[^[:alnum:]]/', '_', strtolower($collection['type']['colour']));
      }
      $dates[$index] = [
        '#theme' => [
          'localgov_waste_collection_collection_item__' . $colour_theme_keys[$collection['type']['colour']],
          'localgov_waste_collection_collection_item',
        ],
        '#date' => $collection['date'],
        '#label' => $collection['type']['label'],
        '#holiday' => $collection['holiday'] ?? FALSE,
        '#colour' => $collection['type']['colour'],
        '#colour_name' => $colour_theme_keys[$collection['type']['colour']],
        '#service_updates_path' => $this->config->get('service_updates_path') ?? '/',
      ];
    }

    $build['#dates'] = $dates;

    if (!empty($collections) && $address = $this->dataProvider->getAddressFromUprn($uprn)) {
      $build['#address'] = $address;
      $build['#uprn'] = $uprn;
      $build['#service_updates_path'] = $this->config->get('service_updates_path') ?? '/';
      $build['#collections'] = $collections;
      if (isset($collections["pdf"])) {
        $build['#pdf'] = $collections["pdf"];
      }
    }
    else {
      $build['#retrievalError'] = $this->t('Collection schedule could not be retrieved.');
    }

    $build['#theme'] = 'localgov_waste_collection_view_schedule';
    return $build;
  }

  /**
   * Download the collection schedule for a UPRN as an iCal file.
   *
   * @param string $uprn
   *   UPRN to lookup.
   *
   * @return \Symfony\Component\HttpFoundation\StreamedResponse
   *   Streamed response
   */
  public function downloadSchedule(string $uprn): StreamedResponse {
    $response = new StreamedResponse();
    $collections = $this->dataProvider->getCollections($uprn);
    $calendar = Calendar::create((string) $this->t('Bin collection dates'));
    if (!empty($collections)) {
      $contentDisposition = $response->headers->makeDisposition(ResponseHeaderBag::DISPOSITION_ATTACHMENT, $uprn . '.ics');
      $response->headers->set('Content-Type', 'text/calendar; charset=utf-8');
      $response->headers->set('Content-Disposition', $contentDisposition);
      $response->setCallback(function () use ($uprn, &$collections, $calendar): void {
        date_default_timezone_set('Europe/London');

        // Using UPRN for the event uid means users can add multiple properties,
        // and refresh an existing calendar without adding dupes.
        foreach ($collections["dates"] as $collection) {
          $event = Event::create(Unicode::ucfirst((string) $this->t('@colour lidded bins for @label', [
            '@colour' => $collection["type"]["colour"],
            '@label' => $collection["type"]["label"],
          ])))
            ->startsAt(new \DateTime($collection["date"] . ' 06:00:00'))
            ->uniqueIdentifier($uprn . '-' . $collection["date"] . '-' . $collection["type"]["colour"])
            ->description((string) $this->t('Please make sure you have your bin or sacks out by 7:00am ready for collection.'))
            ->alertMinutesBefore(
              720,
              (string) $this->t('@label bin collection tomorrow - have your bins or sacks out by 7:00am', ['@label' => $collection["type"]["label"]]));
          $calendar->event($event);
        }

        // Include a reminder in the ics file to re-download the schedule.
        $lastCollection = end($collections["dates"]);
        $reminderDate = new \DateTime($lastCollection["date"] . ' 18:00:00');
        $reminderDate->modify('-2 weeks');
        $calendarLink = Url::fromRoute('localgov_waste_collection.download', ['uprn' => $uprn], ['absolute' => 'true'])
          ->toString();
        $event = Event::create((string) $this->t('Download your bin collection calendar'))
          ->startsAt($reminderDate)
          ->uniqueIdentifier($uprn . '-reminder-' . $lastCollection["date"])
          ->description((string) $this->t('Your bin collection calendar information expires in 14 days. Go to @calendarlink to update your calendar.', ['@calendarlink' => $calendarLink]))
          ->alertAt(
            $reminderDate,
            (string) $this->t('Update your bin collection calendar')
          );
        $calendar->event($event);

        echo $calendar->get();
      });
    }
    return $response;
  }

}
