<?php

declare(strict_types=1);

namespace Drupal\localgov_waste_collection\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * {@inheritdoc}
 */
class PostcodeForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'localgov_waste_collection_postcode_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $form['#attributes']['class'][] = 'waste-collection-form';

    $form['postcode'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Postcode'),
      '#description' => $this->t('For example @sample-postcode', [
        '@sample-postcode' => $this->config('localgov_waste_collection.settings')->get('sample_postcode'),
      ]),
      '#description_display' => 'before',
      '#maxlength' => 8,
      '#size' => 40,
      '#weight' => '0',
      '#attributes' => [
        'class' => ['waste-postcode'],
      ],
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Find'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    if (!$this->isValidPostcode($form_state->getValue('postcode'))) {
      $form_state->setErrorByName('postcode', $this->t('Please enter a valid postcode.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $form_state->setRedirect('localgov_waste_collection.find', ['postcode' => urlencode(trim($form_state->getValue('postcode')))]);
  }

  /**
   * Basic postcode validation, catch early before API call.
   *
   * @param string $postcode
   *   Postcode.
   *
   * @return bool
   *   True if valid.
   */
  public function isValidPostcode($postcode): bool {
    // Remove all whitespace.
    $postcode = preg_replace('/\s/', '', $postcode);
    // Uppercase.
    $postcode = strtoupper($postcode);

    return preg_match("/^[A-Z]{1,2}[0-9]{2,3}[A-Z]{2}$/", $postcode)
      || preg_match("/^[A-Z]{1,2}[0-9]{1}[A-Z]{1}[0-9]{1}[A-Z]{2}$/", $postcode)
      || preg_match("/^GIR0[A-Z]{2}$/", $postcode);
  }

}
