<?php

declare(strict_types=1);

namespace Drupal\localgov_waste_collection_whitespace_provider\Plugin\DataProvider;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\localgov_waste_collection\DataProviderBase;
use Drupal\localgov_waste_collection\Exception\PostcodeSearchException;
use Drupal\localgov_waste_collection_whitespace_provider\Exception\WhitespaceException;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\BadResponseException;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Psr7\Response;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Whitespace waste collection data provider.
 *
 * @DataProvider(
 *   id = "whitespace_data_provider",
 *   label = @Translation("Whitespace Data Provider")
 * )
 */
class WhitespaceDataProvider extends DataProviderBase implements ContainerFactoryPluginInterface {

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The http client.
   *
   * @var \GuzzleHttp\Client
   */
  protected $httpClient;

  /**
   * The logger channel factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerChannelFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $container->get('config.factory'),
      $container->get('http_client'),
      $container->get('logger.factory'),
    );
  }

  /**
   * Creates a WhitespaceDataProvider instance.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \GuzzleHttp\Client $http_client
   *   The default http client.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_channel_factory
   *   The logger channel factory.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    Client $http_client,
    LoggerChannelFactoryInterface $logger_channel_factory,
  ) {
    $this->configFactory = $config_factory;
    $this->httpClient = $http_client;
    $this->loggerChannelFactory = $logger_channel_factory;
  }

  /**
   * Makes a SOAP call to the Whitespace endpoint.
   *
   * @param string $operation
   *   The name of the SOAP operation, e.g. 'GetCollectionByUprnAndDate'.
   * @param string $soap_body
   *   The body of the SOAP call.
   *
   * @return \GuzzleHttp\Psr7\Response
   *   The response from Whitespace.
   *
   * @throws \Drupal\localgov_waste_collection_whitespace_provider\Exception\WhitespaceException
   *   Throws an exception if there is a problem with the request. The caller
   *   should log this with appropriate context (such as request parameters).
   */
  protected function makeSoapCall(string $operation, string $soap_body): Response {
    $xml_request = $this->getSoapHeader() . $soap_body . $this->getSoapFooter();

    $options = [
      'body' => $xml_request,
      'headers' => [
        "SOAPAction" => "http://webservices.whitespacews.com/" . $operation,
        "Content-Type" => "text/xml",
        "accept" => "*/*",
        "accept-encoding" => "gzip, deflate",
      ],
    ];

    try {
      $response = $this->httpClient->request(
        'POST',
        $this->configFactory->get('localgov_waste_collection_whitespace_provider.settings')->get('endpoint') . 'WSAPIService.svc/' . $operation,
        $options
      );
    }
    catch (BadResponseException $guzzle_response_exception) {
      // Catch a problem with the request of response.
      $response = $guzzle_response_exception->getResponse();
      $soap_body = $response->getBody()->getContents();

      // Get the message from Whitespace's SOAP response.
      $xml = simplexml_load_string($soap_body);
      $xml->registerXPathNamespace('soap', 'http://schemas.xmlsoap.org/soap/envelope/');
      foreach ($xml->xpath('//faultstring') as $item) {
        $error_message = (string) $item;
      }

      throw new WhitespaceException($error_message, 0, $guzzle_response_exception);
    }
    catch (RequestException $guzzle_exception) {
      // Catch other Guzzle exceptions.
      throw new WhitespaceException($guzzle_exception->getMessage(), 0, $guzzle_exception);
    }

    return $response;
  }

  /**
   * Construct SOAP header XML.
   */
  private function getSoapHeader():string {
    $config = \Drupal::config('localgov_waste_collection_whitespace_provider.settings');
    return '<soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"
                  xmlns:web="http://webservices.whitespacews.com/">
    <soapenv:Header>
        <wsse:Security soapenv:mustUnderstand="1"
                       xmlns:wsse="http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd"
                       xmlns:wsu="http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd">
            <wsse:UsernameToken>
                <wsse:Username>' . $config->get('username') . '</wsse:Username>
                <wsse:Password
                        Type="http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-username-token-profile-1.0#PasswordText">' . $config->get('password') . '</wsse:Password>
            </wsse:UsernameToken>
        </wsse:Security>
    </soapenv:Header>';
  }

  /**
   * Construct SOAP footer XML.
   */
  private function getSoapFooter():string {
    return '</soapenv:Envelope>';
  }

  /**
   * {@inheritdoc}
   */
  public function findAddressesByPostcode(string $postcode): array {
    $properties = [];

    $soapBody = '
    <soapenv:Body>
        <web:GetAddresses>
            <web:getAddressInput>
                <web:Postcode> ' . $postcode . '</web:Postcode>
                <web:IncludeCommercialSites>0</web:IncludeCommercialSites>
            </web:getAddressInput>
        </web:GetAddresses>
    </soapenv:Body>
    ';

    try {
      $response = $this->makeSoapCall('GetAddresses', $soapBody);
    }
    catch (WhitespaceException $e) {
      $this->loggerChannelFactory->get('localgov_waste_collection_whitespace_provider')->error("Whitespace error for waste collection search for postcode '%postcode'. Error message was: " . $e->getMessage(), [
        '%postcode' => $postcode,
      ]);

      throw new PostcodeSearchException($e->getMessage(), 0, $e);
    }

    $xml = simplexml_load_string((string) $response->getBody());

    $addressesXml = $xml
      ->children('s', TRUE)
      ->Body
      ->children('http://webservices.whitespacews.com/')
      ->GetAddressesResponse
      ->GetAddressesResult
      ->Addresses;

    $json = json_encode($addressesXml);
    $addresses = json_decode($json, TRUE);

    if ($addresses) {
      if (!$addresses["Address"][0]) {
        $properties[$addresses["Address"]["AccountSiteUprn"]] = $this->formatAddress($addresses["Address"]);
      }
      else {
        foreach ($addresses["Address"] as $address) {
          $properties[$address["AccountSiteUprn"]] = $this->formatAddress($address);
        }
      }
      asort($properties, SORT_NUMERIC);
    }

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function getAddressFromUprn(string $uprn): ?string {
    $address = NULL;

    $soapBody = '
    <soapenv:Body>
        <web:GetSiteInfo>
            <web:siteInfoInput>
                <web:Uprn> ' . $uprn . '</web:Uprn>
            </web:siteInfoInput>
        </web:GetSiteInfo>
    </soapenv:Body>
    ';

    try {
      $response = $this->makeSoapCall('GetSiteInfo', $soapBody);
    }
    catch (WhitespaceException $e) {
      $this->loggerChannelFactory->get('localgov_waste_collection_whitespace_provider')->error('Error getting addresses from UPRN %uprn', [
        '%uprn' => $uprn,
      ]);

      return NULL;
    }

    $xml = simplexml_load_string((string) $response->getBody());

    $addressXml = $xml
      ->children('s', TRUE)
      ->Body
      ->children('http://webservices.whitespacews.com/')
      ->GetSiteInfoResponse
      ->GetSiteInfoResult
      ->Site
      ->Site;

    $json = json_encode($addressXml);
    $address = $this->formatAddress(json_decode($json, TRUE));

    return $address;
  }

  /**
   * Format raw address.
   */
  private function formatAddress(array $address):string {
    $addressElements = [];
    $address = array_filter($address);
    if (isset($address["SiteName"])) {
      $addressElements[] = (string) $address["SiteName"];
    }
    if (isset($address["SiteAddressNumber"])) {
      $addressElements[] = str_replace('  ', ' ', $address["SiteAddressNumber"] . ' ' . ($address["SiteAddress1"] ?? '') . ' ' . $address["SiteAddress2"]);
    }
    if (isset($address["SiteTown"])) {
      $addressElements[] = (string) $address["SiteTown"];
    }
    $addressElements[] = (string) $address["SitePostCode"];

    return implode(', ', $addressElements);
  }

  /**
   * Get waste collections for a specified UPRN.
   *
   * @return array
   *   An array of waste collections.
   */
  public function getCollections(string $uprn): array {
    // Temporary solution for API v11.0 - date range collections call not
    // available until v11.1, so we loop per month.
    // Call GetCollectionByUprnAndDate six times and merge results.
    $collections = [];
    $currentDate = new \DateTime();

    for ($monthOffset = 0; $monthOffset < 6; $monthOffset++) {
      $callDate = $currentDate->format('01/m/Y');
      $currentDate->modify('first day of next month');

      $soapBody = '
      <soapenv:Body>
        <web:GetCollectionByUprnAndDate>
          <web:getCollectionByUprnAndDateInput>
            <web:Uprn>' . $uprn . '</web:Uprn>
            <web:NextCollectionFromDate>' . $callDate . '</web:NextCollectionFromDate>
          </web:getCollectionByUprnAndDateInput>
        </web:GetCollectionByUprnAndDate>
      </soapenv:Body>
      ';

      try {
        $response = $this->makeSoapCall('GetCollectionByUprnAndDate', $soapBody);
      }
      catch (WhitespaceException $e) {
        $this->loggerChannelFactory->get('localgov_waste_collection_whitespace_provider')->error('Error getting collections from UPRN %uprn', [
          '%uprn' => $uprn,
        ]);
      }

      $xml = simplexml_load_string((string) $response->getBody());

      // Check the error code and return an empty array if there are no
      // collections found.
      $error_code = (string) $xml
        ->children('s', TRUE)
        ->Body
        ->children('http://webservices.whitespacews.com/')
        ->GetCollectionByUprnAndDateResponse
        ->GetCollectionByUprnAndDateResult
        ->ErrorCode;

      if ($error_code != 0) {
        return [
          'dates' => [],
        ];
      }

      $collectionsXml = $xml
        ->children('s', TRUE)
        ->Body
        ->children('http://webservices.whitespacews.com/')
        ->GetCollectionByUprnAndDateResponse
        ->GetCollectionByUprnAndDateResult
        ->Collections;

      $json = json_encode($collectionsXml);
      $monthCollections = json_decode($json, TRUE);

      foreach ($monthCollections["Collection"] as $collection) {
        $collections["Collection"][] = $collection;
      }
    }
    return $this->getCollectionsArray($collections);
  }

  /**
   * Construct an array of collections.
   */
  private function getCollectionsArray(array $collections): array {
    // Get the bin types from config.
    $collection_services = $this->configFactory->get('localgov_waste_collection_whitespace_provider.settings')->get('collection_services');
    $collection_services = array_column($collection_services, NULL, 'name');

    $collectionArray = [
      'weekly_collection' => [],
      'dates' => [],
    ];

    foreach ($collections["Collection"] as $collection) {
      // Replace date slashes with hyphens so date sorting will not assume US
      // format.
      $formattedDate = str_replace('/', '-', substr($collection["Date"], 0, 10));
      $holiday = NULL;
      $type = $collection_services[$collection["Service"]];
      $collectionArray["dates"][] = [
        "date" => $formattedDate,
        "bin" => $collection["Service"],
        "holiday" => $holiday,
        "type" => $type,
      ];
    }
    usort($collectionArray["dates"], static fn($a, $b) => strtotime($a["date"]) - strtotime($b["date"]));
    $collectionArray["dates"] = $this->removeDuplicateCollections($collectionArray["dates"]);
    return $collectionArray;
  }

  /**
   * Gets the definitions of collection services from Whitespace.
   *
   * @return array
   *   An array of the service definitions, keyed by the service name.
   *
   * @throws \Drupal\localgov_waste_collection_whitespace_provider\Exception\WhitespaceException
   *   Throws an exception if there is a problem with the request. The caller
   *   should log this with appropriate context.
   */
  public function getCollectionServices(): array {
    $soap_body = '
      <soapenv:Body>
        <web:GetServices>
          <web:serviceInput>
            <web:IncludeCommercial>0</web:IncludeCommercial>
          </web:serviceInput>
        </web:GetServices>
      </soapenv:Body>
      ';

    $response = $this->makeSoapCall('GetServices', $soap_body);
    $xml = simplexml_load_string((string) $response->getBody());

    // Check the error code and return an empty array if there are no
    // services found.
    $error_code = (string) $xml
      ->children('s', TRUE)
      ->Body
      ->children('http://webservices.whitespacews.com/')
      ->GetServicesResponse
      ->GetServicesResult
      ->ErrorCode;

    if ($error_code != 0) {
      $error_message = (string) $xml
        ->children('s', TRUE)
        ->Body
        ->children('http://webservices.whitespacews.com/')
        ->GetServicesResponse
        ->GetServicesResult
        ->CodeDescription;

      throw new WhitespaceException('Error getting services from Whitespace, with message: ' . $error_message);
    }

    $services_xml = $xml
      ->children('s', TRUE)
      ->Body
      ->children('http://webservices.whitespacews.com/')
      ->GetServicesResponse
      ->GetServicesResult
      ->Services;

    $service_definitions = [];
    foreach ($services_xml->children() as $service) {
      if ((string) $service->ServiceIsCollection == 'false') {
        continue;
      }
      // @todo Is there anything else we should filter out?
      $service = (array) $service;

      $service = array_filter($service, fn ($s) => !is_object($s));

      $service_definitions[$service['ServiceName']] = $service;
    }

    return $service_definitions;
  }

  /**
   * Remove duplicates from the collection dates array.
   *
   * @param array $array
   *   Raw data from Whitespace.
   *
   * @return array
   *   De-duped data.
   */
  public function removeDuplicateCollections($array): array {
    $uniqueCollections = array_map("unserialize", array_unique(array_map("serialize", $array)));

    foreach ($uniqueCollections as $key => $value) {
      if (is_array($value)) {
        $uniqueCollections[$key] = self::removeDuplicateCollections($value);
      }
    }

    return $uniqueCollections;
  }

}
