<?php

declare(strict_types=1);

namespace Drupal\localgov_waste_collection_csv_provider\Batch;

use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Database\Database;
use Drupal\Core\Database\DatabaseExceptionWrapper;
use Drupal\Core\Render\Markup;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Methods for running the CSV import in a batch.
 *
 * @package Drupal\csvimport
 */
class CsvImportBatch {

  use StringTranslationTrait;

  /**
   * Handle batch completion.
   *
   * Creates a new CSV file containing all failed rows if any.
   */
  public static function csvimportImportFinished($success, $results, $operations): TranslatableMarkup {
    $messenger = \Drupal::messenger();
    $translator = \Drupal::translation();

    if (!empty($results['rows_imported'])) {
      $messenger->addMessage(
        $translator->translate(
          '@rows_processed @row_type processed (includes count of any failed rows).',
          [
            '@rows_processed' => $results["rows_imported"],
            '@row_type' => $results["row_type"],
          ]
        ),
        MessengerInterface::TYPE_STATUS
      );
    }

    if (!empty($results['failed_rows'])) {
      $failedRows = '';
      if (count($results["failed_rows"]) > 50) {
        $failedRows = $translator->translate('More than 50 rows failed, please check source file.');
      }
      else {
        foreach ($results['failed_rows'] as $row) {
          $failedRows .= implode(', ', $row) . '<br>';
        }
      }

      $messenger->addMessage(
        Markup::create($translator->translate(
          '@failed_row_count row@plural failed to import:<br><br>@failed_rows',
          [
            '@failed_row_count' => count($results["failed_rows"]),
            '@plural' => count($results["failed_rows"]) > 1 ? 's' : '',
            '@failed_rows' => $failedRows,
          ]
        )),
        MessengerInterface::TYPE_ERROR
      );
    }

    return $translator->translate('The CSV import has completed.');
  }

  /**
   * Remember the uploaded CSV filename.
   */
  public static function csvimportRememberFilename($filename, &$context): void {
    $context['results']['uploaded_filename'] = $filename;
  }

  /**
   * Process a single property file line.
   */
  public static function csvimportImportPropertyLine($line, &$context): void {
    $translator = \Drupal::translation();
    $context['results']['row_type'] = 'properties';
    $context['results']['rows_imported'] = ($context['results']['rows_imported'] ?? 0) + 1;
    $line = array_map('base64_decode', $line);

    $context['message'] = $translator->translate('Importing UPRN %uprn', ['%uprn' => $line[0]]);

    $connection = Database::getConnection();

    if ($line[1] === '0' || $line[3] === 'Monthly') {
      $context['results']['failed_rows'][] = $line;
    }
    else {
      try {
        $connection->upsert('localgov_waste_collection_properties')
          ->fields([
            'uprn' => $line[0],
            'postcode' => str_replace(' ', '', $line[1]),
            'address' => $line[2],
            'updated' => time(),
          ])
          ->key('uprn')
          ->execute();
      }
      catch (DatabaseExceptionWrapper $exception) {
        $context['results']['failed_rows'][] = $line;
      }
    }
  }

  /**
   * Process a single collections file line.
   */
  public static function csvimportImportCollectionLine($line, &$context): void {
    $translator = \Drupal::translation();
    $context['results']['row_type'] = 'collections';
    $context['results']['rows_imported'] = ($context['results']['rows_imported'] ?? 0) + 1;
    $line = array_map('base64_decode', $line);

    $context['message'] = $translator->translate('Importing row %row', ['%row' => $context['results']['rows_imported']]);

    $connection = Database::getConnection();

    try {
      $connection->insert('localgov_waste_collection_collections')
        ->fields([
          'collection_round' => $line[0],
          'waste_type' => $line[1],
          'bin_colour' => $line[2],
          'collection_date' => $line[3],
        ])
        ->execute();
    }
    catch (DatabaseExceptionWrapper $exception) {
      $context['results']['failed_rows'][] = $line;
    }
  }

  /**
   * Process a single property / round mapping file line.
   */
  public static function csvimportImportPropertyRoundLine($line, &$context): void {
    $translator = \Drupal::translation();
    $context['results']['row_type'] = 'property/round';
    $context['results']['rows_imported'] = ($context['results']['rows_imported'] ?? 0) + 1;
    $line = array_map('base64_decode', $line);

    $context['message'] = $translator->translate('Importing row %row', ['%row' => $context['results']['rows_imported']]);

    $connection = Database::getConnection();

    try {
      $connection->insert('localgov_waste_collection_property_round')
        ->fields([
          'uprn' => $line[0],
          'collection_round' => $line[1],
        ])
        ->execute();
    }
    catch (DatabaseExceptionWrapper $exception) {
      $context['results']['failed_rows'][] = $line;
    }
  }

}
