<?php

declare(strict_types=1);

namespace Drupal\localgov_waste_collection_csv_provider\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\File\Exception\FileException;
use Drupal\Core\File\FileExists;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\File\FileSystem;
use Drupal\file\Entity\File;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Waste collection CSV upload Form.
 */
class WasteCSVForm extends FormBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The file_system service.
   *
   * @var \Drupal\Core\File\FileSystem
   */
  protected FileSystem $fileSystem;

  public function __construct(Connection $database, FileSystem $fileSystem) {
    $this->database = $database;
    $this->fileSystem = $fileSystem;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('database'),
      $container->get('file_system')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'localgov_waste_collection_csv_form';
  }

  /**
   * Content Import Form.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {

    $form['property_import_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Property import mode'),
      '#options' => ['replace' => $this->t('Clear and import'), 'update' => $this->t('Append and update')],
      '#default_value' => 'replace',
      '#description' => $this->t('How should the properties in the CSV file be processed?'),
    ];

    $form['property_csv_file'] = [
      '#type' => 'file',
      '#title' => $this->t('Property CSV file'),
      '#description' => $this->t('CSV file containing address data for all properties.'),
    ];

    $form['collection_csv_file'] = [
      '#type' => 'file',
      '#title' => $this->t('Collection rounds CSV file'),
      '#description' => $this->t('CSV file containing data for collection rounds.'),
    ];

    $form['property_round_csv_file'] = [
      '#type' => 'file',
      '#title' => $this->t('Property to collection round mapping CSV file'),
      '#description' => $this->t('CSV file mapping UPRNs to collection rounds.'),
    ];

    $form['pdf_file'] = [
      '#type' => 'file',
      '#title' => $this->t('Waste collection round PDF file'),
      '#description' => $this->t('A PDF file for a waste collection round. Must be named correctly, case sensitive, using the collection_round value from the property CSV file.'),
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Upload and process'),
      '#button_type' => 'primary',
    ];

    return $form;

  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {

    $allFiles = $this->getRequest()->files->get('files', []);

    if (isset($allFiles['pdf_file'])) {

      $filePath = $allFiles['pdf_file']->getRealPath();
      $destinationFile = 'public://' . $allFiles['pdf_file']->getClientOriginalName();

      try {
        $this->fileSystem->copy($filePath, $destinationFile, FileExists::Replace);
        $file = File::create([
          'filename' => $allFiles['pdf_file']->getClientOriginalName(),
          'uri' => $destinationFile,
          'status' => 1,
          'uid' => 1,
        ]);
        $file->save();
        $this->messenger()->addMessage($this->t('PDF @file_name uploaded successfully.', ['@file_name' => $allFiles['pdf_file']->getClientOriginalName()]));
      }
      catch (FileException $e) {
        $this->messenger()->addError($this->t('Error uploading file'));
        die();
      }

    }

    if (isset($allFiles['property_csv_file'])) {

      $file = $allFiles['property_csv_file'];
      $filePath = $file->getRealPath();
      $destinationFile = 'temporary://waste_collection.csv';

      try {
        $this->fileSystem->copy($filePath, $destinationFile, FileExists::Replace);
      }
      catch (FileException $e) {
        $this->messenger()->addError($this->t('Error uploading file'));
        die();
      }

      $batch = [
        'title' => $this->t('Importing properties'),
        'operations' => [],
        'init_message' => $this->t('Commencing'),
        'progress_message' => $this->t('Processed @current out of @total.'),
        'error_message' => $this->t('An error occurred during property import.'),
        'finished' => '\Drupal\localgov_waste_collection_csv_provider\Batch\CsvImportBatch::csvimportImportFinished',
      ];

      $processFile = $this->fileSystem->realpath($destinationFile);

      if ($handle = fopen($processFile, 'rb')) {

        $batch['operations'][] = [
          '\Drupal\localgov_waste_collection_csv_provider\Batch\CsvImportBatch::csvimportRememberFilename',
          [$processFile],
        ];

        if ($form_state->getValue('active_data_provider') === 'replace') {
          $this->database->truncate('localgov_waste_collection_properties')->execute();
        }

        while ($line = fgetcsv($handle, 4096)) {

          $batch['operations'][] = [
            '\Drupal\localgov_waste_collection_csv_provider\Batch\CsvImportBatch::csvimportImportPropertyLine',
            [array_map('base64_encode', $line)],
          ];
        }

        fclose($handle);
      }

      batch_set($batch);
    }

    if (isset($allFiles['collection_csv_file'])) {

      $file = $allFiles['collection_csv_file'];
      $filePath = $file->getRealPath();
      $destinationFile = 'temporary://waste_collection.csv';

      try {
        $this->fileSystem->copy($filePath, $destinationFile, FileExists::Replace);
      }
      catch (FileException $e) {
        $this->messenger()->addError($this->t('Error uploading file'));
        die();
      }

      $batch = [
        'title' => $this->t('Importing collection rounds'),
        'operations' => [],
        'init_message' => $this->t('Commencing'),
        'progress_message' => $this->t('Processed @current out of @total.'),
        'error_message' => $this->t('An error occurred during collection round import.'),
        'finished' => '\Drupal\localgov_waste_collection_csv_provider\Batch\CsvImportBatch::csvimportImportFinished',
      ];

      $processFile = $this->fileSystem->realpath($destinationFile);

      if ($handle = fopen($processFile, 'rb')) {

        $batch['operations'][] = [
          '\Drupal\localgov_waste_collection_csv_provider\Batch\CsvImportBatch::csvimportRememberFilename',
          [$processFile],
        ];

        $this->database->truncate('localgov_waste_collection_collections')->execute();

        while ($line = fgetcsv($handle, 4096)) {

          $batch['operations'][] = [
            '\Drupal\localgov_waste_collection_csv_provider\Batch\CsvImportBatch::csvimportImportCollectionLine',
            [array_map('base64_encode', $line)],
          ];
        }

        fclose($handle);
      }

      batch_set($batch);
    }

    if (isset($allFiles['property_round_csv_file'])) {

      $file = $allFiles['property_round_csv_file'];
      $filePath = $file->getRealPath();
      $destinationFile = 'temporary://waste_collection.csv';

      try {
        $this->fileSystem->copy($filePath, $destinationFile, FileExists::Replace);
      }
      catch (FileException $e) {
        $this->messenger()->addError($this->t('Error uploading file'));
        die();
      }

      $batch = [
        'title' => $this->t('Importing property / round mapping'),
        'operations' => [],
        'init_message' => $this->t('Commencing'),
        'progress_message' => $this->t('Processed @current out of @total.'),
        'error_message' => $this->t('An error occurred during property/round import.'),
        'finished' => '\Drupal\localgov_waste_collection_csv_provider\Batch\CsvImportBatch::csvimportImportFinished',
      ];

      $processFile = $this->fileSystem->realpath($destinationFile);

      if ($handle = fopen($processFile, 'rb')) {

        $batch['operations'][] = [
          '\Drupal\localgov_waste_collection_csv_provider\Batch\CsvImportBatch::csvimportRememberFilename',
          [$processFile],
        ];

        $this->database->truncate('localgov_waste_collection_property_round')->execute();

        while ($line = fgetcsv($handle, 4096)) {

          $batch['operations'][] = [
            '\Drupal\localgov_waste_collection_csv_provider\Batch\CsvImportBatch::csvimportImportPropertyRoundLine',
            [array_map('base64_encode', $line)],
          ];
        }

        fclose($handle);
      }

      batch_set($batch);
    }
  }

}
