<?php

declare(strict_types=1);

namespace Drupal\localgov_waste_collection\Service;

use Drupal\Component\DependencyInjection\ContainerInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use GuzzleHttp\Client;

/**
 * Public Holidays provider.
 */
class PublicHolidays {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The configuration object.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected Config $config;

  /**
   * The cache.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cache;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache.
   */
  public function __construct(ConfigFactoryInterface $configFactory, CacheBackendInterface $cache) {
    $this->configFactory = $configFactory;
    $this->config = $this->configFactory->get('localgov_waste_collection.settings');
    $this->cache = $cache;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('cache.default'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getPublicHolidays(): array {
    $publicHolidays = [];
    $ignoredHolidays = [];

    $ignoredHolidaysSource = $this->config->get('ignored_holidays');
    if ($ignoredHolidaysSource !== '') {
      $ignoredHolidays = explode("\r\n", $ignoredHolidaysSource);
    }

    $endPoint = 'https://www.gov.uk/bank-holidays.json';

    $cacheId = 'localgov_waste_collection_holidays';
    if ($cache = $this->cache->get($cacheId)) {
      return $cache->data;
    }

    $region = $this->config->get('region');
    $client = new Client();
    $response = $client
      ->get($endPoint, ['headers' => ['Accept' => 'application/json']]);
    $allHolidays = Json::decode($response->getBody()->getContents());
    if (isset($allHolidays[$region]["events"])) {
      foreach ($allHolidays[$region]["events"] as $holiday) {
        if (strtotime($holiday["date"]) > (time() - 86400)) {
          $publicHolidays[date('d-m-Y', strtotime($holiday["date"]))] = $holiday["title"];
        }
      }
    }

    $publicHolidays = array_diff($publicHolidays, $ignoredHolidays);
    $this->cache->set($cacheId, $publicHolidays, time() + 86400);
    return $publicHolidays;
  }

}
