<?php

declare(strict_types=1);

namespace Drupal\localgov_waste_collection_csv_provider\Plugin\DataProvider;

use Drupal\localgov_waste_collection\DataProviderBase;

/**
 * CSV waste collection data provider.
 *
 * @DataProvider(
 *   id = "csv_data_provider",
 *   label = @Translation("CSV Data Provider")
 * )
 */
class CsvDataProvider extends DataProviderBase {

  /**
   * {@inheritdoc}
   */
  public function findAddressesByPostcode(string $postcode): array {
    $searchPostcode = str_replace(' ', '', $postcode);
    $connection = \Drupal::database();
    $query = $connection->select('localgov_waste_collection_properties', 'l');
    $query->condition('l.postcode', $searchPostcode);
    $query->fields('l', ['uprn', 'address']);
    // Had an issue getting MySQL REGEXP to id if the address started with a
    // number, so this is hacky:
    $numbered = $query->addExpression("LEFT(address,1) IN ('0','1','2','3','4','5','6','7','8','9')", 'numbered');
    $address_number = $query->AddExpression("CAST(SUBSTRING_INDEX(l.address, ',', 1) AS SIGNED INTEGER)", 'address_number');
    $query->orderBy($numbered, 'DESC');
    $query->orderBy($address_number);
    $query->orderBy('l.address');
    return $query->execute()->fetchAllKeyed(0, 1);
  }

  /**
   * {@inheritdoc}
   */
  public function getAddressFromUprn(string $uprn): ?string {
    $address = NULL;

    $connection = \Drupal::database();
    $query = $connection->select('localgov_waste_collection_properties', 'l');
    $query->condition('l.uprn', $uprn);
    $query->fields('l', ['address']);
    // Use fetchAllKeyed.
    $address = $query->execute()->fetch();

    return $address->address;
  }

  /**
   * Parse waste collections.
   *
   * @return array
   *   An array of waste collection dates.
   */
  private function parseCollections(array $collectionSource): array {
    $collections = ["dates" => []];

    foreach ($collectionSource as $collection) {
      $collections["dates"][] = [
        "date" => $collection['collection_date'],
        "type" => [
          "label" => $collection['waste_type'],
          "colour" => $collection['bin_colour'],
        ],
      ];
    }

    return $collections;
  }

  /**
   * Get waste collections for a specified UPRN.
   *
   * @return array
   *   An array of waste collections.
   */
  public function getCollections(string $uprn): array {
    $collections = [];
    $collections['dates'] = [];
    $connection = \Drupal::database();

    $query = $connection->select('localgov_waste_collection_property_round', 'pr')
      ->fields('pr', ['collection_round'])
      ->condition('pr.uprn', $uprn);
    $rounds = $query->execute()->fetchCol();

    $pdfs = [];

    foreach ($rounds as $roundCode) {
      // Fetch future collections for this round.
      $roundCollections = $this->getRoundCollections($roundCode);
      $collections['dates'] = array_merge($collections['dates'], $this->parseCollections($roundCollections)['dates']);

      // Check for a round-specific PDF.
      $pdfPath = 'public://' . $roundCode . '.pdf';
      if (file_exists($pdfPath)) {
        $pdfs[] = $pdfPath;
      }
    }

    // Sort the combined collection dates by date ascending.
    usort($collections['dates'], function ($a, $b) {
      return strcmp($a['date'], $b['date']);
    });

    if (!empty($pdfs)) {
      $collections['pdf'] = count($pdfs) === 1 ? $pdfs[0] : $pdfs;
    }

    return $collections;
  }

  /**
   * Get the collection dates for a specified round.
   *
   * @param string $roundCode
   *   The collection round identifier.
   *
   * @return array
   *   An array of collection date data.
   */
  private function getRoundCollections(string $roundCode): array {
    $connection = \Drupal::database();
    $query = $connection->select('localgov_waste_collection_collections', 'c')
      ->fields('c', ['collection_date', 'waste_type', 'bin_colour'])
      ->condition('c.collection_round', $roundCode)
      ->condition('c.collection_date', date('Y-m-d'), '>=');
    return $query->execute()->fetchAll(\PDO::FETCH_ASSOC);
  }

}
