<?php

declare(strict_types=1);

namespace Drupal\localgov_waste_collection\EventSubscriber;

use Drupal\Core\Routing\CurrentRouteMatch;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\localgov_core\Event\PageHeaderDisplayEvent;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Set page header.
 *
 * @package Drupal\localgov_waste_collection\EventSubscriber
 */
class PageHeaderSubscriber implements EventSubscriberInterface {
  use StringTranslationTrait;

  /**
   * The route matching service.
   *
   * @var \\Drupal\Core\Routing\CurrentRouteMatch
   */
  protected $currentRouteMatch;

  /**
   * Constructor.
   */
  public function __construct(CurrentRouteMatch $currentRouteMatch) {
    $this->currentRouteMatch = $currentRouteMatch;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('current_route_match')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      PageHeaderDisplayEvent::EVENT_NAME => ['setPageHeader', 0],
    ];
  }

  /**
   * Set page title and lede.
   */
  public function setPageHeader(PageHeaderDisplayEvent $event): void {
    $route_name = $this->currentRouteMatch->getRouteName();
    if ($route_name && str_contains($route_name, 'localgov_waste_collection')) {
      switch ($route_name) {
        case 'localgov_waste_collection.view':
          $lede = $this->t('Bin collection days for this address');
          break;

        case 'localgov_waste_collection.form':
          $lede = $this->t('Enter a postcode to find out when your bins will be collected.');
          break;

        case 'localgov_waste_collection.find':
          $lede = $this->t('Select your address to find out when your bins will be collected');
          break;

        default:
          $lede = $this->t('Find your bin collection schedule');
      }

      $event->setLede([
        '#type' => 'html_tag',
        '#tag' => 'p',
        '#value' => $lede,
      ]);
    }
  }

}
