<?php

declare(strict_types=1);

namespace Drupal\localgov_waste_collection_whitespace_provider\Form;

use Drupal\Component\Utility\Html;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\localgov_waste_collection_whitespace_provider\Exception\WhitespaceException;

/**
 * Waste collection Whitespace configuration Form.
 */
class WhitespaceSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'localgov_waste_collection_whitespace_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['localgov_waste_collection_whitespace_provider.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('localgov_waste_collection_whitespace_provider.settings');

    $form['endpoint'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Endpoint'),
      '#default_value' => $config->get('endpoint') ?? '',
      '#description' => $this->t('The full Whitespace endpoint URL including trailing slash, in the format: https://xxx.whitespacews.com/'),
      '#required' => TRUE,
    ];

    $form['username'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Username'),
      '#default_value' => $config->get('username') ?? '',
      '#description' => $this->t('The username for your Whitespace API access.'),
      '#required' => TRUE,
    ];

    $form['password'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Password'),
      '#default_value' => $config->get('password') ?? '',
      '#description' => $this->t('The password for your Whitespace API access.'),
      '#required' => TRUE,
    ];

    // Stored config for services.
    $collection_services_config = $config->get('collection_services') ?? [];
    // Service definitions from Whitespace, stored in form state from clicking
    // the 'Fetch' button.
    $collection_service_definitions = $form_state->get('collection_services');

    $wrapper_id = Html::getUniqueId('localgov_waste_collection_whitespace_provider-collection-services-wrapper');

    $form['collection_services'] = [
      '#type' => 'fieldset',
      '#tree' => TRUE,
      '#title' => $this->t('Collection services'),
      '#description_display' => 'before',
      '#description' => $this->t("Set the label and bin colour for each collection service defined in Whitespace. Leave a label blank to use the name from Whitespace."),
      '#prefix' => '<div id="' . $wrapper_id . '">',
      '#suffix' => '</div>',
    ];

    $form['collection_services']['fetch_service_names'] = [
      '#type' => 'submit',
      '#value' => $this->t('Fetch service names from Whitespace'),
      '#submit' => [
        '::submitFetchServiceNames',
      ],
      '#ajax' => [
        'callback' => '::ajaxFetchServiceNames',
        'wrapper' => $wrapper_id,
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Fetching data from Whitespace...'),
        ],
      ],
      // Set weights for that the messages go just above the button, as
      // otherwise they are potentially off-screen and not noticeable.
      '#weight' => 200,
    ];

    // Disable the button if config has not been saved yet. It's oo complicated
    // to have the form make this functionality available when the
    // credentials values have been entered but not yet saved, because our
    // data provider plugin expects to read credentials from config.
    if (!$config->get('username')) {
      $form['collection_services']['fetch_service_names']['#disabled'] = TRUE;

      $form['collection_services']['#description'] = $this->t('Save the settings with the Whitespace access credentials to fetch collection service names.');
    }

    // Change the button label if we already have config.
    if ($collection_services_config) {
      $form['collection_services']['fetch_service_names']['#value'] = $this->t('Update service names from Whitespace');
    }

    if ($collection_services_config || $collection_service_definitions) {
      // Build form elements for config first.
      foreach ($collection_services_config as $collection_services_config_item) {
        $collection_service_name = $collection_services_config_item['name'];

        $form['collection_services']['services'][$collection_service_name] = [
          '#type' => 'fieldset',
          '#title' => $this->t("Collection service '@service'", [
            '@service' => $collection_service_name,
          ]),
        ];
        $form['collection_services']['services'][$collection_service_name]['label'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Label'),
          '#default_value' => $collection_services_config_item['label'],
        ];
        $form['collection_services']['services'][$collection_service_name]['colour'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Bin colour'),
          '#default_value' => $collection_services_config_item['colour'],
        ];
      }

      // If there are Whitespace service definitions from an AJAX call to fetch
      // them, use those to refine the form.
      if (!empty($collection_service_definitions)) {
        foreach ($collection_service_definitions as $collection_service_name => $collection_service_definition_item) {
          // Add a form element for the service if it's not already there. This
          // covers the case where new services have been added to Whitespace
          // since the config was saved.
          if (!isset($form['collection_services']['services'][$collection_service_name])) {
            $form['collection_services']['services'][$collection_service_name] = [
              '#type' => 'fieldset',
              '#title' => $this->t("Collection service '@service'", [
                '@service' => $collection_service_name,
              ]),
            ];
            $form['collection_services']['services'][$collection_service_name]['label'] = [
              '#type' => 'textfield',
              '#title' => $this->t('Label'),
              // Default to the Whitespace service name.
              '#default_value' => $collection_service_name,
            ];
            $form['collection_services']['services'][$collection_service_name]['colour'] = [
              '#type' => 'textfield',
              '#title' => $this->t('Bin colour'),
            ];
          }
        }

        // Remove services from the form which are no longer in Whitespace.
        $form['collection_services']['services'] = array_intersect_key($form['collection_services']['services'], $collection_service_definitions);
      }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * Submit handler for the fetch services button.
   */
  public function submitFetchServiceNames(array &$form, FormStateInterface $form_state): void {
    $whitespace_data_provider = \Drupal::service('plugin.manager.localgov_waste_collection.data_provider')->createInstance('whitespace_data_provider');

    try {
      $collection_services = $whitespace_data_provider->getCollectionServices();
      $form_state->set('collection_services', $collection_services);
    }
    catch (WhitespaceException $e) {
      $this->messenger()->addError((string) $this->t('There was a problem connecting to Whitespace.'));
    }

    $form_state->setRebuild();
  }

  /**
   * AJAX callback for the fetch services button.
   */
  public function ajaxFetchServiceNames(array &$form, FormStateInterface $form_state) {
    // Add the messages. (According docs they should show -- rabbit hole I don't
    // have time for, and this way we get to weight them how we want.)
    $form['collection_services']['messages'] = [
      '#type' => 'status_messages',
      '#weight' => 100,
    ];

    return $form['collection_services'];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('localgov_waste_collection_whitespace_provider.settings');

    $config->set('endpoint', $form_state->getValue('endpoint'))
      ->set('username', $form_state->getValue('username'))
      ->set('password', $form_state->getValue('password'));

    $collection_service_values = [];
    foreach ($form_state->getValue(['collection_services', 'services'], []) as $whitespace_name => $collection_service_value) {
      // Skip an array with only empty values.
      if (empty(array_filter($collection_service_value))) {
        continue;
      }

      // Set the Whitespace name into the value.
      $collection_service_value['name'] = $whitespace_name;

      $collection_service_values[] = $collection_service_value;
    }
    $config->set('collection_services', $collection_service_values);

    $config->save();

    parent::submitForm($form, $form_state);
  }

}
