<?php

declare(strict_types=1);

namespace Drupal\localgov_waste_collection\Form;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\localgov_waste_collection\DataProviderManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Waste collection schedule settings for this site.
 */
final class WasteCollectionSettingsForm extends ConfigFormBase {

  /**
   * The data provider manager.
   *
   * @var \Drupal\localgov_waste_collection\DataProviderManager
   */
  private $dataProviderManager;

  /**
   * The cache.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  private CacheBackendInterface $cache;

  /**
   * Constructs a new SettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\localgov_waste_collection\DataProviderManager $dataProviderManager
   *   The data provider manager.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache..
   */
  public function __construct(ConfigFactoryInterface $config_factory, DataProviderManager $dataProviderManager, CacheBackendInterface $cache) {
    parent::__construct($config_factory);

    $this->dataProviderManager = $dataProviderManager;
    $this->cache = $cache;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('plugin.manager.localgov_waste_collection.data_provider'),
      $container->get('cache.default'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'localgov_waste_collection_waste_collection_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['localgov_waste_collection.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('localgov_waste_collection.settings');

    // Get available plugins.
    $plugin_definitions = $this->dataProviderManager->getDefinitions();
    $options = [];

    foreach ($plugin_definitions as $id => $definition) {
      $options[$id] = $definition['label'];
    }

    $form['active_data_provider'] = [
      '#type' => 'select',
      '#title' => $this->t('Data Provider'),
      '#options' => $options,
      '#required' => TRUE,
      '#default_value' => $config->get('active_data_provider'),
      '#description' => $this->t('Select a waste collection schedule data provider to use.'),
    ];

    $form['sample_postcode'] = [
      '#type' => 'textfield',
      '#title' => $this->t("Sample postcode"),
      '#default_value' => $config->get('sample_postcode'),
      '#description' => $this->t("The example postcode to show in the form."),
    ];

    $form['waste_collection_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Base path'),
      '#default_value' => $config->get('waste_collection_path'),
      '#description' => $this->t('Base path for waste collection schedule lookup.'),
      '#required' => TRUE,
    ];

    $form['waste_collection_page_titles'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Page titles'),
      '#tree' => TRUE,
    ];

    $form['waste_collection_page_titles']['form_pages'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Form pages title'),
      '#default_value' => $config->get('waste_collection_page_titles.form_pages'),
      '#description' => $this->t('The page title for the postcode form and address find form pages.'),
    ];

    $form['waste_collection_page_titles']['results_page'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Collection results page title'),
      '#default_value' => $config->get('waste_collection_page_titles.results_page'),
      '#description' => $this->t('The page title for the waste collections results page.'),
    ];

    $form['public_holidays'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Public holidays'),
      '#description' => $this->t('Public holidays can be downloaded automatically from gov.uk, and a notice displayed on collections that fall on holidays. Source: <a href="https://www.gov.uk/bank-holidays.json">https://www.gov.uk/bank-holidays.json</a>'),
      '#description_display' => 'before',
    ];

    $form['public_holidays']['public_holidays'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable public holidays'),
      '#default_value' => $config->get('public_holidays'),
    ];

    $form['public_holidays']['region'] = [
      '#type' => 'select',
      '#title' => $this->t('Region'),
      '#options' => [
        'england-and-wales' => $this->t('England and Wales'),
        'northern-ireland' => $this->t('Northern Ireland'),
        'scotland' => $this->t('Scotland'),
      ],
      '#default_value' => $config->get('region'),
      '#description' => $this->t("Select which region's holidays to use."),
      '#states' => [
        'visible' => [
          ':input[name="public_holidays"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['public_holidays']['service_updates_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Path or URL for service updates'),
      '#default_value' => $config->get('service_updates_path'),
      '#description' => $this->t('The path or URL for service update information during public holidays.'),
      '#states' => [
        'visible' => [
          ':input[name="public_holidays"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['public_holidays']['ignored_holidays'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Ignored holidays'),
      '#default_value' => $config->get('ignored_holidays'),
      '#description' => $this->t('A list of holidays to be ignored, one per line. See <a href="https://www.gov.uk/bank-holidays.json">https://www.gov.uk/bank-holidays.json</a> for supported values.'),
      '#states' => [
        'visible' => [
          ':input[name="public_holidays"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['ics_event_title_format'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Calendar event title format'),
      '#description' => $this->t('Format for calendar event titles. Use @time, @colour and @label as placeholders.'),
      '#default_value' => $config->get('ics_event_title_format') ?: '@colour lidded bins for @label',
      '#required' => TRUE,
    ];

    $form['ics_event_description_format'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Calendar event description format'),
      '#description' => $this->t('Format for calendar event descriptions. Use @time, @colour and @label as placeholders.'),
      '#default_value' => $config->get('ics_event_description_format') ?: 'Please make sure you have your bin or sacks out by @time ready for collection.',
      '#required' => TRUE,
    ];

    $form['ics_event_alert_format'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Calendar event alert format'),
      '#description' => $this->t('Format for calendar event alert sent 12 hours before collection time. Use @time, @colour and @label as placeholders.'),
      '#default_value' => $config->get('ics_event_alert_format') ?: '@label bin collection tomorrow - have your bins or sacks out by @time',
      '#required' => TRUE,
    ];

    $form['collection_time'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Collection start time'),
      '#description' => $this->t('The time when collection events start in calendar entries and on the schedule page. Format: HH:MM:SS (24-hour format).'),
      '#default_value' => $config->get('collection_time') ?: '06:00:00',
      '#required' => TRUE,
      '#size' => 10,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('localgov_waste_collection.settings')
      ->set('active_data_provider', $form_state->getValue('active_data_provider'))
      ->set('sample_postcode', $form_state->getValue('sample_postcode'))
      ->set('waste_collection_path', $form_state->getValue('waste_collection_path'))
      ->set('waste_collection_page_titles.form_pages', $form_state->getValue(
        ['waste_collection_page_titles', 'form_pages']
      ))
      ->set('waste_collection_page_titles.results_page', $form_state->getValue(
        ['waste_collection_page_titles', 'results_page']
      ))
      ->set('public_holidays', $form_state->getValue('public_holidays'))
      ->set('service_updates_path', $form_state->getValue('service_updates_path'))
      ->set('region', $form_state->getValue('region'))
      ->set('ignored_holidays', $form_state->getValue('ignored_holidays'))
      ->set('ics_event_title_format', $form_state->getValue('ics_event_title_format'))
      ->set('ics_event_description_format', $form_state->getValue('ics_event_description_format'))
      ->set('ics_event_alert_format', $form_state->getValue('ics_event_alert_format'))
      ->set('collection_time', $form_state->getValue('collection_time'))
      ->save();
    $this->cache->invalidate('localgov_waste_collection_holidays');

    parent::submitForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $custom_route = $form_state->getValue('waste_collection_path');

    // Ensure the path starts with a forward slash.
    if (empty($custom_route) || !str_starts_with($custom_route, '/')) {
      $form_state->setErrorByName('waste_collection_path', $this->t('The path must start with a forward slash (/)'));
      return;
    }

    // Check iCal event time format.
    $collection_time = $form_state->getValue('collection_time');
    if (!preg_match('/^([01]?[0-9]|2[0-3]):[0-5][0-9]:[0-5][0-9]$/', $collection_time)) {
      $form_state->setErrorByName('collection_time', $this->t('Collection time must be in HH:MM:SS format (24-hour).'));
    }
  }

}
