<?php

namespace Drupal\localized_config\Config;

use Drupal\Core\Config\ConfigCrudEvent;
use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Language\LanguageDefault;
use Drupal\language\Config\LanguageConfigFactoryOverride;
use Drupal\language\Config\LanguageConfigFactoryOverrideInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Makes the LanguageConfigFactoryOverride friendlier to Localized Config.
 */
class LocalizedConfigFactoryOverrideDecorator extends LanguageConfigFactoryOverride {

  /**
   * Intercepts all method calls not defined in the decorator.
   *
   * @param string $method
   *   Name of the method.
   * @param array|mixed $args
   *   Arguments directed to the method.
   *
   * @return mixed
   *   The results of the method call.
   */
  public function __call($method, $args): mixed {
    return call_user_func_array([$this->innerService, $method], $args);
  }

  /**
   * Constructs the LanguageConfigFactoryDecorator object.
   *
   * @param \Drupal\language\Config\LanguageConfigFactoryOverrideInterface $innerService
   *   The original service.
   * @param \Drupal\Core\Config\StorageInterface $storage
   *   The configuration storage engine.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   An event dispatcher instance to use for configuration events.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config
   *   The typed configuration manager.
   * @param \Drupal\Core\Language\LanguageDefault $default_language
   *   The default language.
   * @param array|null $defaultLanguageValues
   *   Default language values.
   * @param bool|null $translateEnglish
   *   Whether to treat English as a translatable language.
   */
  public function __construct(
    protected LanguageConfigFactoryOverrideInterface $innerService,
    StorageInterface $storage,
    EventDispatcherInterface $event_dispatcher,
    TypedConfigManagerInterface $typed_config,
    LanguageDefault $default_language,
    protected ?array $defaultLanguageValues,
    protected ?bool $translateEnglish = TRUE,
  ) {
    parent::__construct($storage,
      $event_dispatcher,
      $typed_config,
      $default_language,
      $defaultLanguageValues,
      $translateEnglish,
    );
  }

  /**
   * Prevents Localized Config overrides from being deleted with the original.
   *
   * @param \Drupal\Core\Config\ConfigCrudEvent $event
   *   The config CRUD event.
   */
  public function onConfigSave(ConfigCrudEvent $event): void {
    $config = $event->getConfig();
    $name = $config->getName();

    if (str_starts_with($name, 'localized_config')) {
      return;
    }

    foreach (\Drupal::languageManager()->getLanguages() as $language) {
      $config_translation = $this->getOverride($language->getId(), $name);
      if (!$config_translation->isNew()) {
        $this->filterOverride($config, $config_translation);
      }
    }
  }

}
