<?php

namespace Drupal\localized_config;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\CacheTagsInvalidator;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;

/**
 * Helper for locale settings/variable-related methods.
 *
 * @package Drupal\localized_config
 */
class LocalizedConfigHelper {

  /**
   * Are languages supported for this system?
   *
   * @var bool
   */
  protected $languagesSupported;

  /**
   * LocalizedConfigHelper constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config
   *   Config Factory service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   Language Manager service.
   * @param \Drupal\localized_config\LocalizedConfigPluginManager $pluginManager
   *   Localized Config Plugin Manager service.
   * @param \Drupal\Core\Cache\CacheTagsInvalidator $cacheTagsInvalidator
   *   Cache Tags invalidator service.
   * @param \Drupal\localized_config\LocalizedConfigLanguageHelper $languageHelper
   *   Localized Config language-related methods.
   */
  public function __construct(protected ConfigFactoryInterface $config, protected LanguageManagerInterface $languageManager, protected LocalizedConfigPluginManager $pluginManager, protected CacheTagsInvalidator $cacheTagsInvalidator, protected LocalizedConfigLanguageHelper $languageHelper) {
    $this->languagesSupported = (bool) $this->config->get('localized_config.settings')->get('enable_languages');
  }

  /**
   * Returns whether languages are supported on this system.
   *
   * @return bool
   *   The result.
   */
  public function languagesSupported(): bool {
    return $this->languagesSupported;
  }

  /**
   * Get localized variables from all configs.
   *
   * @param string|object|null|false $language
   *   The language to get the config from.
   *
   * @return mixed
   *   Return either the value, or NULL if none found.
   */
  public function getAllVariables($language = NULL): mixed {
    $variables = [];

    $plugin_definitions = $this->pluginManager->getDefinitions();
    foreach ($plugin_definitions as $plugin_id => $plugin) {
      $variables[$plugin_id] = $this->getVariable($plugin_id, NULL, $language);
    }

    return $variables;
  }

  /**
   * Get a localized variable from the configs.
   *
   * @param string $plugin_name
   *   The Localized Config plugin which the config belongs to.
   * @param string|null $var_name
   *   The id of the value to fetch.
   * @param mixed $language
   *   The language to get the config from.
   *
   * @return mixed
   *   Return either the value, or NULL if none found.
   *
   * @throws \Exception
   */
  public function getVariable(string $plugin_name, $var_name = NULL, $language = NULL): mixed {
    $language_manager = $this->languageManager;
    $original_language = $language_manager->getConfigOverrideLanguage();
    $language_manager->setConfigOverrideLanguage($original_language);
    $config = $this->config->get('localized_config.' . $plugin_name);

    // Ensure config exists.
    if (empty($config)) {
      throw new \Exception("Localized Config: There is no config with the plugin name '$plugin_name'");
    }

    if ($language === FALSE) {

      // Fetch the original ('global') variable.
      return $config->getOriginal($var_name, FALSE);
    }

    // Fetch the language.
    $language = $this->getApplicableLanguageObject($language);
    $language_manager->setConfigOverrideLanguage($language);

    // We are getting the config again, because it has just been overridden.
    $config = $this->config->get('localized_config.' . $plugin_name);

    // If blacklisted the config will default to global.
    if (!empty($language) && $this->languageHelper->isBlacklistedLanguage($language->getId())) {
      // Fetch the original ('global') variable.
      return $config->getOriginal($var_name, FALSE);
    }

    // Fetch the variable in the specified language.
    $variable = $config->get($var_name);

    if (!isset($variable) || !$variable) {

      // Fetch the original ('global') variable.
      $variable = $config->getOriginal($var_name, FALSE);
    }

    return $variable;
  }

  /**
   * Fetch the global value of a certain variable.
   *
   * @param string $plugin_name
   *   Name of the plugin to fetch from.
   * @param string|null $var_name
   *   Name of the variable to fetch.
   *
   * @return mixed
   *   The global variable.
   */
  public function getGlobalVariable(string $plugin_name, $var_name): mixed {
    return $this->getVariable($plugin_name, $var_name, FALSE);
  }

  /**
   * Fetch an array of variables from a set of languages.
   *
   * @param string $plugin_name
   *   Name of the plugin to fetch from.
   * @param string|null $var_name
   *   Name of the variable to fetch.
   * @param array[LanguageInterface] $languages
   *   An array of languages to fetch the variables from, keyed by langcode.
   *
   * @return array
   *   An array of variables, keyed by langcode.
   */
  public function getVariableFromLanguages(string $plugin_name, $var_name = NULL, array $languages = []): array {
    if (empty($languages)) {
      $language_manager = $this->languageManager;
      $languages = $language_manager->getLanguages();
    }
    $result = [];
    foreach ($languages as $langcode => $language) {
      $result[$langcode] = $this->getVariable($plugin_name, $var_name, $language);
    }

    return $result;
  }

  /**
   * Checks whether a given Localized Config plugin is enabled.
   *
   * @param string $plugin_name
   *   Which config to fetch a value from?
   * @param string|object|null $language
   *   The language to get the config from.
   *
   * @return bool
   *   Returns true or false.
   */
  public function pluginEnabled(string $plugin_name, $language = NULL): bool {
    return (bool) $this->getVariable($plugin_name, 'enabled', $language);
  }

  /**
   * Provides a list of all languages for which a certain plugin is enabled.
   *
   * @param string $plugin_name
   *   Which config to fetch the enabled state for?
   *
   * @return \Drupal\Core\Language\LanguageInterface[]
   *   An array of language objects, keyed by langcode.
   */
  public function getEnabledLanguages($plugin_name): array {
    $language_manager = $this->languageManager;
    $languages = $language_manager->getLanguages();
    $result = [];
    foreach ($languages as $langcode => $language) {
      if ($this->pluginEnabled($plugin_name, $language)) {
        $result[$langcode] = $language;
      }
    }

    return $result;
  }

  /**
   * Receives langcode or language object and returns an object.
   *
   * @param string|\Drupal\Core\Language\LanguageInterface|null $language
   *   The language to be tested.
   *
   * @return \Drupal\Core\Language\LanguageInterface|FALSE
   *   The language, or FALSE if no such language present.
   */
  protected function getApplicableLanguageObject($language): LanguageInterface|false {
    $language_manager = $this->languageManager;
    if ($language instanceof LanguageInterface) {
      return $language;
    }

    $cache = &drupal_static(__FUNCTION__ . ':' . ($language ?? 'null'));
    if ($cache) {
      return $cache;
    }

    if ($language) {
      if (\is_string($language)) {
        $language_object = $language_manager->getLanguage($language);
        if ($language_object) {

          $cache = $language_object;
          return $language_object;
        }
      }
    }

    $language = $language_manager->getCurrentLanguage(LanguageInterface::TYPE_CONTENT);
    $cache = $language;

    return $language;
  }

  /**
   * Adds cache information about a Localized Config to something.
   *
   * @param string $plugin_name
   *   Which config to apply cache information from?
   * @param array|CacheableMetadata $target
   *   The target config or render array.
   */
  public function addConfigCache($plugin_name, array|CacheableMetadata &$target): void {
    if ($target instanceof CacheableMetadata) {
      $metadata = $target;
    }
    else {
      $metadata = CacheableMetadata::createFromRenderArray($target);
    }

    $metadata->addCacheTags([
      "localized_config.{$plugin_name}",
    ]);

    if (\is_array($target)) {
      $metadata->applyTo($target);
    }
  }

  /**
   * Invalidates the cache of a certain Localized Config.
   *
   * @param string $plugin_name
   *   Which config to invalidate the cache for?
   */
  public function invalidateConfigCache($plugin_name): void {
    $this->cacheTagsInvalidator->invalidateTags(["localized_config.{$plugin_name}"]);
  }

}
