<?php

namespace Drupal\localizejs\Plugin\Block;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a block with links to switch language.
 *
 * @Block(
 *   id = "localizejs_language_switcher",
 *   admin_label = @Translation("Localize language switcher"),
 * )
 */
class LanguageSwitcher extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The current path.
   *
   * @var string
   */
  protected $currentPath;

  /**
   * Whether to use a different domain for each language or not.
   *
   * @var boolean
   */
  protected $useDomains;

  /**
   * Constructs a Drupal\localizejs\Plugin\Block\LanguageSwitcher object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory service.
   * @param string $current_path
   *   The current request path.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ConfigFactoryInterface $config_factory, $current_path) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->configFactory = $config_factory;
    $this->currentPath = $current_path;
    $this->useDomains = !empty($this->configFactory->get('localizejs.settings')->get('use_domains'));
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $config_factory = $container->get('config.factory');
    $current_path = $container->get('path.current')->getPath();
    return new static($configuration, $plugin_id, $plugin_definition, $config_factory, $current_path);
  }

  /**
   * Builds an array with data of each language from configuration.
   *
   * @return array
   *   Each item is a language info array with code, label and domain as keys.
   */
  protected function getLanguagesFromConfig() {
    $languages = [];
    $language_list = $this->configFactory->listAll('localizejs.language.');
    foreach($language_list as $language_config_name) {
      $language_config = $this->configFactory->get($language_config_name);
      $data = [
        'code' => $language_config->get('code'),
        'label' => $language_config->get('label'),
        'domain' => $language_config->get('domain'),
      ];
      if (!isset($languages[$language_config->get('weight')])) {
        $languages[$language_config->get('weight')] = $data;
      }
      else {
        $languages[] = $data;
      }
    }
    ksort($languages);
    return $languages;
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $languages = $this->getLanguagesFromConfig();
    $block = [
      '#theme' => 'localize_language_switcher',
      '#languages' => $languages,
      '#current_path' => $this->currentPath,
      '#use_domains' => $this->useDomains,
      '#attached' => [
        'library' => ['localizejs/language_switcher']
      ],
    ];

    $block['#attached']['drupalSettings']['localizejs']['language_switcher'] = [
      'languages' => $languages,
      'useDomains' => $this->useDomains,
    ];

    return $block;
  }

  /**
   * {@inheritdoc}
   */
  protected function blockAccess(AccountInterface $account) {
    return AccessResult::allowedIfHasPermission($account, 'access content');
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge() {
    return 0;
  }

}
