<?php

namespace Drupal\log_cache_tags\Hook;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;

/**
 * Hook implementations for the Log Cache Tags module.
 */
class LogCacheTagsHooks {

  use StringTranslationTrait;

  public function __construct(
    protected StateInterface $state,
    TranslationInterface $string_translation,
  ) {
    $this->setStringTranslation($string_translation);
  }

  /**
   * Implements hook_help().
   */
  #[Hook('help')]
  public function help($route_name, RouteMatchInterface $route_match): string {
    $output = '';

    if ($route_name == 'help.page.log_cache_tags') {
      $output = '<h2>' . t('About') . '</h2>';
      $output .= '<p>' . t('The Log Cache Tags module logs the cache tags that are invalidated. It should only be enabled on development environments and never on production.') . '</p>';
    }

    return $output;
  }

  /**
   * Implements hook_form_FORM_ID_alter() for \Drupal\system\Form\DevelopmentSettingsForm.
   */
  #[Hook('form_development_settings_form_alter')]
  public function developmentSettingsFormAlter(array &$form, FormStateInterface $form_state): void {
    $form['log_invalidated_tags'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log the invalidated cache tags'),
      '#default_value' => $this->state->get('log_cache_tags_log_invalidated_tags', FALSE),
      '#weight' => 3,
    ];
    $form['actions']['#weight'] = 6;
    $form['actions']['submit']['#submit'][] = [self::class, 'developmentSettingsSubmit'];
  }

  /**
   * Form submission handler for \Drupal\system\Form\DevelopmentSettingsForm.
   *
   * @see self::developmentSettingsFormAlter()
   */
  public static function developmentSettingsSubmit(array &$form, FormStateInterface $form_state): void {
    \Drupal::state()
      ->set('log_cache_tags_log_invalidated_tags', $form_state->getValue('log_invalidated_tags'));
  }

}
