<?php

namespace Drupal\Tests\logged_in\Kernel;

use Drupal\logged_in\Plugin\views\field\LoggedIn;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\Tests\views\Kernel\ViewsKernelTestBase;
use Drupal\views\ResultRow;
use Drupal\views\Tests\ViewTestData;
use Drupal\views\Views;

/**
 * Tests the functionality of the "Logged In Status" views field.
 *
 * @coversDefaultClass \Drupal\logged_in\Plugin\views\field\LoggedIn
 * @group logged_in
 */
class LoggedInFieldTest extends ViewsKernelTestBase {

  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['user', 'logged_in', 'logged_in_test_views'];

  /**
   * {@inheritdoc}
   */
  public static $testViews = ['test_logged_in'];

  /**
   * {@inheritdoc}
   */
  protected function setUp($import_test_views = TRUE): void {
    parent::setUp(FALSE);

    // Install the user entity schema.
    $this->installEntitySchema('user');

    // Install user module configuration.
    $this->installConfig(['user']);

    // Create a test user to have data in the view.
    $this->createUser();

    // Disable strict config schema checking for this test.
    $this->strictConfigSchema = FALSE;

    // Import test views from our test module.
    if ($import_test_views) {
      ViewTestData::createTestViews(static::class, ['logged_in_test_views']);
    }
  }

  /**
   * Test the plugin directly without Views complexity.
   */
  public function testFieldPluginDirectly(): void {
    // Test 1: Anonymous user.
    $current_user = \Drupal::currentUser();
    $this->assertFalse($current_user->isAuthenticated(), 'Should start anonymous.');

    $plugin = \Drupal::service('plugin.manager.views.field')->createInstance('logged_in', []);
    $row = new ResultRow();

    $anonymous_result = $plugin->getValue($row);
    $this->assertEquals('', $anonymous_result, 'Plugin should return empty string for anonymous user.');

    // Test 2: Authenticated user.
    $authenticated_user = $this->createUser();
    $this->setCurrentUser($authenticated_user);

    // Create fresh plugin instance after user change.
    $plugin_auth = \Drupal::service('plugin.manager.views.field')->createInstance('logged_in', []);
    $authenticated_result = $plugin_auth->getValue($row);
    $this->assertEquals('1', $authenticated_result, 'Plugin should return "1" for authenticated user.');
  }

  /**
   * Tests the field output for anonymous users.
   */
  public function testLoggedInFieldAnonymous(): void {
    // Ensure we start as anonymous.
    $current_user = \Drupal::currentUser();
    $this->assertFalse($current_user->isAuthenticated(), 'Should be anonymous user.');

    $view = Views::getView('test_logged_in');
    $view->setDisplay();
    $view->initHandlers();
    $this->executeView($view);

    $this->assertGreaterThan(0, count($view->result), 'View should have results.');
    $row = $view->result[0];
    $field = $view->field['logged_in'];

    $anonymous_value = $field->getValue($row);
    $this->assertEquals('', $anonymous_value, 'Field should return empty string for anonymous user.');
  }

  /**
   * Tests that the plugin is discoverable.
   */
  public function testPluginDiscovery(): void {
    $plugin_manager = \Drupal::service('plugin.manager.views.field');
    $plugin_definitions = $plugin_manager->getDefinitions();

    $this->assertArrayHasKey('logged_in', $plugin_definitions, 'The logged_in plugin should be discoverable.');

    // Test creating the plugin directly.
    $plugin = $plugin_manager->createInstance('logged_in', []);
    $this->assertInstanceOf(LoggedIn::class, $plugin, 'Plugin should be the correct class.');
  }

  /**
   * Tests the field output for authenticated users.
   */
  public function testLoggedInFieldAuthenticated(): void {
    // Set up authenticated user from the start.
    $authenticated_user = $this->createUser();
    $this->setCurrentUser($authenticated_user);

    // Verify user context.
    $current_user = \Drupal::currentUser();
    $this->assertTrue($current_user->isAuthenticated(), 'Current user should be authenticated.');

    $view = Views::getView('test_logged_in');
    $view->setDisplay();
    $view->initHandlers();
    $this->executeView($view);

    $this->assertGreaterThan(0, count($view->result), 'View should have results.');
    $row = $view->result[0];
    $field = $view->field['logged_in'];

    $authenticated_value = $field->getValue($row);
    $this->assertEquals('1', $authenticated_value, 'Field should return "1" for authenticated user.');
  }

}
