<?php

namespace Drupal\logger\Plugin\LoggerTarget;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\logger\Attribute\LoggerTarget;
use Drupal\logger\Plugin\LoggerTargetBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a file logger target.
 */
#[LoggerTarget(
  id: 'file',
  label: new TranslatableMarkup('File'),
  description: new TranslatableMarkup('Writes log to a file. Not recommended for production.'),
  weight: 20
)]
class File extends LoggerTargetBase implements ContainerFactoryPluginInterface {

  /**
   * The stream wrapper manager service instance (lazily loaded).
   *
   * @var \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface|null
   */
  protected $streamWrapperManager;

  /**
   * The service container (stored for lazy service access).
   *
   * @var \Symfony\Component\DependencyInjection\ContainerInterface
   */
  protected ContainerInterface $container;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    // Store the container and lazily fetch services when needed.
    $instance->container = $container;
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function persist(string $entry, int $level): void {
    if (!empty($this->configuration['path'] ?? NULL)) {
      // Support any Drupal stream wrapper (public://, private://,
      // temporary://, etc.) for file paths.
      $path = $this->configuration['path'];
      if (str_contains($path, '://')) {
        $path = $this->getStreamWrapperManager()->getViaUri($path)->realpath();
      }
      // @todo User stream wrapper API for file operations.
      file_put_contents($path, $entry . "\n", FILE_APPEND);
    }
  }

  /**
   * Lazily get the stream wrapper manager service.
   *
   * @return \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface
   *   The stream wrapper manager service.
   */
  protected function getStreamWrapperManager() {
    if (!$this->streamWrapperManager) {
      $this->streamWrapperManager = $this->container->get('stream_wrapper_manager');
    }
    return $this->streamWrapperManager;
  }

  /**
   * {@inheritdoc}
   */
  public function validateTarget(array $target): bool {
    return !empty($target['path']) && is_string($target['path']);
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultConfiguration(): array {
    return [
      'destination' => '/tmp/drupal.log',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfigForm() {
    $form['path'] = [
      '#type' => 'textfield',
      '#title' => 'Path',
      '#default_value' => $this->configuration['path'],
      '#required' => TRUE,
      '#description' => $this->t('Put an absolute or relative path to the log file. Relative path will be resolved to the Drupal root. You can use Drupal stream wrappers, for example, <code>temporary://drupal.log</code> or <code>private://log.jsonl</code>.'),
    ];
    return $form;
  }

}
