<?php

namespace Drupal\logger;

/**
 * The Logger entry object, containing all data that will be logged.
 *
 * No declared properties listed, because the data can be structured in a free
 * form, and will be converted to JSON.
 *
 * @see Drupal\logger\Form\SettingsForm::LOGGER_FIELDS for the list of
 * common possible values.
 */
class LoggerEntry implements LoggerEntryInterface {

  /**
   * A storage of the log entry data.
   *
   * @var array
   */
  protected array $data;

  /**
   * {@inheritdoc}
   */
  public function __construct(?array $data = NULL) {
    $this->data = is_array($data)
      ? $data
      : [];
  }

  /**
   * {@inheritdoc}
   */
  public function set(string $key, $value): LoggerEntry {
    $this->data[$key] = $value;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function get(string $key): mixed {
    return $this->data[$key] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function delete(string $key): LoggerEntry {
    unset($this->data[$key]);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setData(array $data): LoggerEntry {
    $this->data = $data;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getData(): array {
    return $this->data;
  }

  /**
   * {@inheritdoc}
   */
  public function cleanEmptyValues(): void {
    $this->data = array_filter($this->data, function ($value) {
      return $value !== NULL && $value !== '' && $value !== [];
    });
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty(): bool {
    return empty($this->data);
  }

  /**
   * {@inheritdoc}
   */
  public function __toString(): string {
    return json_encode($this->data);
  }

}
