<?php

namespace Drupal\logger\Plugin\LoggerTarget;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\logger\Attribute\LoggerTarget;
use Drupal\logger\Plugin\LoggerTargetBase;

/**
 * Provides an output stream logger target.
 */
#[LoggerTarget(
  id: 'output',
  label: new TranslatableMarkup('Output'),
  description: new TranslatableMarkup('Outputs to stdout or stderr.'),
  weight: 10
)]
class Output extends LoggerTargetBase {

  /**
   * {@inheritdoc}
   */
  public function persist(string $entry, int $level): void {
    file_put_contents('php://' . $this->configuration['stream'], $entry . "\n");
  }

  /**
   * {@inheritdoc}
   */
  public function validateTarget(array $target): bool {
    return !empty($target['stream']) && in_array($target['stream'], ['stdout', 'stderr']);
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultConfiguration(): array {
    return [
      'destination' => 'stderr',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfigForm() {
    $form['stream'] = [
      '#type' => 'select',
      '#title' => 'Stream',
      '#default_value' => $this->configuration['stream'],
      '#required' => TRUE,
      '#description' => $this->t('Choose the output stream for the logs. The <code>stderr</code> is recommended to not mix the standard output with logs, for example, in drush commands.'),
      '#options' => [
        'stderr' => $this->t('stderr: Standard Error'),
        'stdout' => $this->t('stdout: Standard Output'),
      ],
    ];
    return $form;
  }

}
