<?php

namespace Drupal\logger\Form;

use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\RfcLogLevel;
use Drupal\logger\Logger\Logger;
use Drupal\logger\Trait\SettingLabelTrait;
use Psr\Log\LogLevel;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Logger settings for this site.
 */
class SettingsForm extends ConfigFormBase {
  use SettingLabelTrait;

  /**
   * The logger target plugin manager.
   *
   * @var \Drupal\Core\Plugin\DefaultPluginManager
   */
  protected $loggerTargetManager;

  /**
   * A TypedConfigManager.
   *
   * @var \Drupal\Core\Config\TypedConfigManagerInterface
   */
  protected TypedConfigManagerInterface $configTyped;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->configTyped = $container->get('config.typed');
    $instance->loggerTargetManager = $container->get('plugin.manager.logger_target');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'logger_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [Logger::CONFIG_NAME];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(Logger::CONFIG_NAME);
    $this->settingsTyped = $this->configTyped->get(Logger::CONFIG_NAME);

    $enabledFields = $config->get(Logger::CONFIG_KEY_FIELDS) ?? [];

    $form[Logger::CONFIG_KEY_FIELDS_ALL] = [
      '#type' => 'checkbox',
      '#title' => $this->getSettingLabel(Logger::CONFIG_KEY_FIELDS_ALL),
      '#description' => $this->t('Enables adding all fields from the context array to the log entries.'),
      '#config_target' => Logger::CONFIG_NAME . ':' . Logger::CONFIG_KEY_FIELDS_ALL,
    ];

    $enabledPredefinedFields = array_keys(Logger::LOGGER_FIELDS);
    $enabledFieldsFromPredefined = array_intersect($enabledFields, $enabledPredefinedFields);
    $enabledFieldsCustom = array_diff($enabledFields, $enabledFieldsFromPredefined);

    $form[Logger::CONFIG_KEY_FIELDS] = [
      '#type' => 'checkboxes',
      '#title' => $this->getSettingLabel(Logger::CONFIG_KEY_FIELDS),
      '#description' => $this->t('Enable fields which should be present in the log entry.'),
      '#options' => [],
      '#config_target' => Logger::CONFIG_NAME . ':' . Logger::CONFIG_KEY_FIELDS,
      // Use the `#default_value` because need a custom preparation of the
      // values from the configuration.
      '#default_value' => array_merge($enabledFieldsFromPredefined, $enabledFieldsFromPredefined),
      '#states' => [
        'visible' => [
          ':input[name="fields_all"]' => ['checked' => FALSE],
        ],
      ],
    ];
    foreach (Logger::LOGGER_FIELDS as $field => $description) {
      // Use ignore till the https://www.drupal.org/project/coder/issues/3326197
      // is fixed.
      // @codingStandardsIgnoreStart
      $form['fields']['#options'][$field] = "<code>$field</code> - " . $this->t($description);
      // @codingStandardsIgnoreEnd
    }

    $form['fields_custom'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom fields'),
      '#description' => $this->t('A comma separated list of additional fields from the context array to include.'),
      // Use the `#default_value` because need a custom preparation of the
      // values from the configuration.
      '#default_value' => implode(', ', $enabledFieldsCustom),
      '#states' => [
        'visible' => [
          ':input[name="fields_all"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form[Logger::CONFIG_KEY_ENTRY_EXCLUDE_EMPTY] = [
      '#type' => 'checkbox',
      '#title' => $this->getSettingLabel(Logger::CONFIG_KEY_ENTRY_EXCLUDE_EMPTY),
      '#description' => $this->t('Enables excluding empty fields (with NULL values and empty strings, zero numeric values are kept) from the log entries.'),
      '#config_target' => Logger::CONFIG_NAME . ':' . Logger::CONFIG_KEY_ENTRY_EXCLUDE_EMPTY,
    ];

    $form['targets'] = [
      '#type' => 'fieldset',
      '#title' => 'Log targets',
      '#tree' => TRUE,
      '#prefix' => '<div id="targets-wrapper">',
      '#suffix' => '</div>',
    ];

    $targets = $form_state->getValue('targets');
    if ($targets === NULL) {
      $targets = $config->get(Logger::CONFIG_KEY_TARGETS) ?? [];
      // If no targets exist, provide a default one.
      if (empty($targets)) {
        $plugins = $this->loggerTargetManager->getDefinitions();
        $default_plugin = array_key_first($plugins);
        $targets = [
          [
            'plugin' => $default_plugin,
            'log_level' => RfcLogLevel::INFO,
            'configuration' => [],
          ],
        ];
      }
    }

    $plugins = $this->loggerTargetManager->getDefinitions();
    foreach ($targets as $delta => $target) {
      // Skip non-numeric keys (like 'add_target').
      if (!is_numeric($delta)) {
        continue;
      }

      if (!is_array($target['configuration'] ?? [])) {
        $target['configuration'] = json_decode($target['configuration'] ?? '{}', TRUE);
      }

      // Check if plugin was changed via AJAX and reset configuration if needed.
      $current_plugin = $form_state->getValue(['targets', $delta, 'plugin']);
      if ($current_plugin && $current_plugin !== $target['plugin']) {
        $target['plugin'] = $current_plugin;
        $pluginTmpInstance = $this->loggerTargetManager->createInstance($target['plugin']);
        $target['configuration'] = $pluginTmpInstance->getDefaultConfiguration();
      }

      $form['targets'][$delta] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Target @number', ['@number' => (int) $delta + 1]),
        '#attributes' => ['class' => ['target-item']],
        '#collapsible' => FALSE,
        '#collapsed' => FALSE,
      ];
      $form['targets'][$delta]['plugin'] = [
        '#type' => 'select',
        '#title' => 'Target plugin',
        '#options' => $this->getPluginOptions($plugins),
        '#description' => $this->t('Select the type of target to log messages to.'),
        '#default_value' => $target['plugin'] ?? array_key_first($plugins),
        '#ajax' => [
          'callback' => '::pluginChangeCallback',
          'wrapper' => "target-{$delta}-config-wrapper",
          'event' => 'change',
        ],
        '#submit' => ['::pluginChangeSubmit'],
        '#target_delta' => $delta,
      ];
      $form['targets'][$delta]['log_level'] = [
        '#type' => 'select',
        '#title' => 'Log level',
        '#options' => $this->getRfcLogLevels(),
        '#default_value' => $target['log_level'] ?? RfcLogLevel::INFO,
        '#description' => $this->t('Select the minimum log level for this target. Only messages at this level or higher will be logged.'),
      ];
      $form['targets'][$delta]['configuration'] = [
        '#type' => 'container',
        '#prefix' => "<div id=\"target-{$delta}-config-wrapper\">",
        '#suffix' => '</div>',
      ];

      // Ensure we have a plugin to work with (use default if not set).
      $plugin_id = $target['plugin'] ?? array_key_first($plugins);

      if ($plugin_id) {
        try {
          if (!$target['configuration']) {
            $pluginTmpInstance = $this->loggerTargetManager->createInstance($plugin_id);
            $target['configuration'] = $pluginTmpInstance->getDefaultConfiguration();
          }
          $pluginInstance = $this->loggerTargetManager->createInstance($plugin_id, $target['configuration']);
          $form['targets'][$delta]['configuration'] += $pluginInstance->getConfigForm();
        }
        catch (\Exception $e) {
          // If plugin creation fails, show an error message.
          $form['targets'][$delta]['configuration']['error'] = [
            '#markup' => $this->t('Error loading plugin configuration: @error', ['@error' => $e->getMessage()]),
            '#prefix' => '<div class="messages messages--error">',
            '#suffix' => '</div>',
          ];
        }
      }
      $form['targets'][$delta]['remove'] = [
        '#type' => 'submit',
        '#value' => $this->t('Remove Target'),
        '#submit' => ['::removeTargetSubmit'],
        '#ajax' => [
          'callback' => '::removeTargetCallback',
          'wrapper' => 'targets-wrapper',
        ],
        '#target_index' => $delta,
        '#attributes' => ['class' => ['button--danger', 'remove-target-btn']],
        '#weight' => 100,
      ];
    }
    $form['targets']['add_target'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add new target'),
      '#submit' => ['::addTargetSubmit'],
      '#ajax' => [
        'callback' => '::addTargetCallback',
        'wrapper' => 'targets-wrapper',
      ],
      '#attributes' => ['class' => ['button--primary', 'add-target-btn']],
      '#weight' => 1000,
    ];

    $form[Logger::CONFIG_KEY_SERVICE_NAME] = [
      '#type' => 'textfield',
      '#title' => $this->getSettingLabel(Logger::CONFIG_KEY_SERVICE_NAME),
      '#description' => $this->t('The name of the service to identify the log source.'),
      '#config_target' => Logger::CONFIG_NAME . ':' . Logger::CONFIG_KEY_SERVICE_NAME,
    ];

    $form[Logger::CONFIG_KEY_EXCEPTION_BACKLOG_ITEMS_LIMIT] = [
      '#type' => 'number',
      '#title' => $this->getSettingLabel(Logger::CONFIG_KEY_EXCEPTION_BACKLOG_ITEMS_LIMIT),
      '#description' => $this->t('The maximum number of the exception backtrace items to log. Set to empty to disable the limit. Usually the backtrace contains a lot of items, and it is not useful to log all of them. The default value is 8.'),
      '#config_target' => Logger::CONFIG_NAME . ':' . Logger::CONFIG_KEY_EXCEPTION_BACKLOG_ITEMS_LIMIT,
    ];

    $form[Logger::CONFIG_KEY_SKIP_EVENT_DISPATCH] = [
      '#type' => 'checkbox',
      '#title' => $this->getSettingLabel(Logger::CONFIG_KEY_SKIP_EVENT_DISPATCH),
      '#description' => $this->t('If checked, the module will not dispatch events for log entries. This is useful for performance reasons if you do not have any subscribers for the log entry.'),
      '#config_target' => Logger::CONFIG_NAME . ':' . Logger::CONFIG_KEY_SKIP_EVENT_DISPATCH,
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Skip validation if this is an AJAX request for adding/removing targets or
    // changing plugins.
    $triggering_element = $form_state->getTriggeringElement();
    if (
      isset($triggering_element['#submit']) &&
      (in_array('::addTargetSubmit', $triggering_element['#submit']) ||
        in_array('::removeTargetSubmit', $triggering_element['#submit']) ||
        in_array('::pluginChangeSubmit', $triggering_element['#submit']))) {
      return;
    }

    // Apply form state values transformation on the validation step, instead of
    // the submit, because ConfigFormBase::validateForm() requires the values to
    // be valid to store in the configuration.
    $fieldsSelected = array_values(array_filter($form_state->getValue(Logger::CONFIG_KEY_FIELDS), function ($value, $key) {
      return $value != 0;
    }, ARRAY_FILTER_USE_BOTH));
    if ($fieldsCustom = array_filter(array_map('trim', explode(',', $form_state->getValue('fields_custom'))))) {
      $fieldsSelected = array_unique(array_merge($fieldsSelected, $fieldsCustom));
    }
    $form_state->setValue(Logger::CONFIG_KEY_FIELDS, array_values($fieldsSelected));

    // Process targets - remove the 'add_target' element and clean up the array.
    $targets = $form_state->getValue('targets') ?? [];
    if (isset($targets['add_target'])) {
      unset($targets['add_target']);
    }

    // Remove any targets that have the 'remove' element and validate required
    // fields.
    $cleanTargets = [];
    foreach ($targets as $delta => $target) {
      if (isset($target['remove'])) {
        unset($target['remove']);
      }

      $targets[$delta]['configuration'] = json_encode($target['configuration'] ?? NULL);

      // Re-index the targets array to prevent gaps.
      $cleanTargets[] = $target;
    }

    $form_state->setValue('targets', $cleanTargets);

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Skip submission if this is an AJAX request for adding/removing targets or
    // changing plugins.
    $triggering_element = $form_state->getTriggeringElement();
    if (
      isset($triggering_element['#submit']) &&
      (in_array('::addTargetSubmit', $triggering_element['#submit']) ||
        in_array('::removeTargetSubmit', $triggering_element['#submit']) ||
        in_array('::pluginChangeSubmit', $triggering_element['#submit']))) {
      return;
    }

    $targetsConfig = [];
    foreach ($form_state->getValue('targets') ?? [] as $delta => $target) {
      if (!is_numeric($delta)) {
        continue;
      }
      unset($target['remove']);
      $target['configuration'] = json_encode($target['configuration'] ?? []);
      $targetsConfig[$delta] = $target;
    }
    $config = $this->configFactory->getEditable(Logger::CONFIG_NAME);
    $config->set(Logger::CONFIG_KEY_TARGETS, $targetsConfig);
    $config->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Returns a list of available syslog facilities.
   *
   * @return array
   *   A list with a numeric key and a string value of the each facility.
   */
  protected function syslogFacilityList() {
    return [
      LOG_USER => 'LOG_USER',
      LOG_LOCAL0 => 'LOG_LOCAL0',
      LOG_LOCAL1 => 'LOG_LOCAL1',
      LOG_LOCAL2 => 'LOG_LOCAL2',
      LOG_LOCAL3 => 'LOG_LOCAL3',
      LOG_LOCAL4 => 'LOG_LOCAL4',
      LOG_LOCAL5 => 'LOG_LOCAL5',
      LOG_LOCAL6 => 'LOG_LOCAL6',
      LOG_LOCAL7 => 'LOG_LOCAL7',
    ];
  }

  /**
   * Gets the available RFC log levels.
   */
  protected static function getRfcLogLevels(): array {
    return [
      RfcLogLevel::EMERGENCY => LogLevel::EMERGENCY,
      RfcLogLevel::ALERT => LogLevel::ALERT,
      RfcLogLevel::CRITICAL => LogLevel::CRITICAL,
      RfcLogLevel::ERROR => LogLevel::ERROR,
      RfcLogLevel::WARNING => LogLevel::WARNING,
      RfcLogLevel::NOTICE => LogLevel::NOTICE,
      RfcLogLevel::INFO => LogLevel::INFO,
      RfcLogLevel::DEBUG => LogLevel::DEBUG,
    ];
  }

  /**
   * Get formatted options for plugin select dropdown.
   *
   * @param array $plugins
   *   The plugin definitions array.
   *
   * @return array
   *   Formatted options with label and description.
   */
  protected function getPluginOptions(array $plugins): array {
    // Sort plugins by weight before processing.
    uasort($plugins, function ($a, $b) {
      $weight_a = $a['weight'] ?? 0;
      $weight_b = $b['weight'] ?? 0;
      return $weight_a <=> $weight_b;
    });

    $options = [];
    foreach ($plugins as $plugin_id => $plugin_definition) {
      // Handle both annotation-based and array-based plugin definitions.
      $label = $plugin_definition['label'] ?? $plugin_definition['id'] ?? $plugin_id;
      $description = $plugin_definition['description'] ?? '';

      // Convert TranslatableMarkup objects to strings if needed.
      if (is_object($label) && method_exists($label, '__toString')) {
        $label = (string) $label;
      }
      if (is_object($description) && method_exists($description, '__toString')) {
        $description = (string) $description;
      }

      if ($description) {
        $options[$plugin_id] = $this->t('@label: @description', [
          '@label' => $label,
          '@description' => $description,
        ]);
      }
      else {
        $options[$plugin_id] = $label;
      }
    }
    return $options;
  }

  /**
   * Ajax callback to add a new target item.
   */
  public function addTargetCallback(array $form, FormStateInterface $form_state): array {
    return $form['targets'];
  }

  /**
   * Submit handler to add a new target item.
   */
  public function addTargetSubmit(array $form, FormStateInterface $form_state): void {
    // Get current targets from form state.
    $targets = $form_state->getValue('targets') ?? [];

    // Remove the add_target element if it exists.
    if (isset($targets['add_target'])) {
      unset($targets['add_target']);
    }

    // Find the next available index.
    $new_index = 0;
    if (!empty($targets)) {
      $indices = array_keys($targets);
      $new_index = max($indices) + 1;
    }

    // Get available plugins and set the first one as default.
    $plugins = $this->loggerTargetManager->getDefinitions();
    $default_plugin = array_key_first($plugins);

    // Add a new target with default values.
    $targets[$new_index] = [
      'plugin' => $default_plugin,
      'log_level' => RfcLogLevel::INFO,
      'configuration' => [],
    ];

    $form_state->setValue('targets', $targets);
    $form_state->setRebuild(TRUE);
  }

  /**
   * Ajax callback to remove a target item.
   */
  public function removeTargetCallback(array $form, FormStateInterface $form_state): array {
    return $form['targets'];
  }

  /**
   * Submit handler to remove a target item.
   */
  public function removeTargetSubmit(array $form, FormStateInterface $form_state): void {
    $triggering_element = $form_state->getTriggeringElement();
    $target_index = $triggering_element['#target_index'];

    $targets = $form_state->getValue('targets') ?? [];

    // Remove the add_target element if it exists.
    if (isset($targets['add_target'])) {
      unset($targets['add_target']);
    }

    // Remove the specific target.
    unset($targets[$target_index]);

    $form_state->setValue('targets', $targets);
    $form_state->setRebuild(TRUE);
  }

  /**
   * Ajax callback for plugin change.
   */
  public function pluginChangeCallback(array $form, FormStateInterface $form_state): array {
    $triggering_element = $form_state->getTriggeringElement();
    $target_delta = $triggering_element['#target_delta'];

    // Get the new plugin type.
    $new_plugin = $form_state->getValue(['targets', $target_delta, 'plugin']);

    // Build new configuration form.
    $config_form = [
      '#type' => 'container',
      '#prefix' => "<div id=\"target-{$target_delta}-config-wrapper\">",
      '#suffix' => '</div>',
    ];

    if ($new_plugin) {
      try {
        // Create new instance with default configuration.
        $pluginInstance = $this->loggerTargetManager->createInstance($new_plugin);
        $config_form += $pluginInstance->getConfigForm();
      }
      catch (\Exception $e) {
        // If plugin creation fails, show an error message.
        $config_form['error'] = [
          '#markup' => $this->t('Error loading plugin configuration: @error', ['@error' => $e->getMessage()]),
          '#prefix' => '<div class="messages messages--error">',
          '#suffix' => '</div>',
        ];
      }
    }

    return $config_form;
  }

  /**
   * Submit handler for plugin change.
   */
  public function pluginChangeSubmit(array $form, FormStateInterface $form_state): void {
    $triggering_element = $form_state->getTriggeringElement();
    $target_delta = $triggering_element['#target_delta'];

    $targets = $form_state->getValue('targets') ?? [];

    // Reset configuration when plugin changes.
    if (isset($targets[$target_delta])) {
      $new_plugin = $form_state->getValue(['targets', $target_delta, 'plugin']);
      if ($new_plugin) {
        try {
          $pluginInstance = $this->loggerTargetManager->createInstance($new_plugin);
          $targets[$target_delta]['plugin'] = $new_plugin;
          $targets[$target_delta]['configuration'] = $pluginInstance->getDefaultConfiguration();
          $form_state->setValue('targets', $targets);
        }
        catch (\Exception) {
          // Handle case where plugin creation fails - fallback to empty config.
          $targets[$target_delta]['configuration'] = [];
          $form_state->setValue('targets', $targets);
        }
      }
    }

    $form_state->setRebuild(TRUE);
  }

}
