<?php

namespace Drupal\logger;

/**
 * The Logger entry object, containing all data that will be logged.
 *
 * No declared properties listed, because the data can be structured in a free
 * form, and will be converted to JSON.
 *
 * @see Drupal\logger\Form\SettingsForm::LOGGER_FIELDS for the list of
 * common possible values.
 */
class LoggerEntry implements LoggerEntryInterface {

  /**
   * The microtime of the entry creation.
   *
   * @var float
   */
  protected float $microtime;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    protected int $logLevel,
    protected array $data = [],
    ?float $microtime = NULL,
  ) {
    $this->microtime = $microtime ?? (hrtime(TRUE) / 1000000000);
  }

  /**
   * {@inheritdoc}
   */
  public function set(string $key, $value): LoggerEntry {
    $this->data[$key] = $value;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function get(string $key): mixed {
    return $this->data[$key] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getLogLevel(): int {
    return $this->logLevel;
  }

  /**
   * {@inheritdoc}
   */
  public function getMicrotime(): float {
    return $this->microtime;
  }

  /**
   * {@inheritdoc}
   */
  public function delete(string $key): LoggerEntry {
    unset($this->data[$key]);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setData(array $data): LoggerEntry {
    $this->data = $data;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getData(): array {
    return $this->data;
  }

  /**
   * {@inheritdoc}
   */
  public function cleanEmptyValues(): void {
    $this->data = array_filter($this->data, function ($value) {
      return $value !== NULL && $value !== '' && $value !== [];
    });
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty(): bool {
    return empty($this->data);
  }

  /**
   * {@inheritdoc}
   */
  public function __toString(): string {
    return json_encode($this->data);
  }

}
