<?php

namespace Drupal\logger\Plugin\LoggerTarget;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\logger\Attribute\LoggerTarget;
use Drupal\logger\LoggerEntryInterface;
use Drupal\logger\Plugin\LoggerTargetBase;
use Drupal\logger\Util\LoggerUtils;

/**
 * Provides an output stream logger target.
 */
#[LoggerTarget(
  id: 'output',
  label: new TranslatableMarkup('Output'),
  description: new TranslatableMarkup('Outputs to stdout or stderr.'),
  weight: 10
)]
class Output extends LoggerTargetBase {

  const STREAM_STDERR = 'php://stderr';
  const STREAM_STDOUT = 'php://stdout';
  const STREAM_CONTAINER_STDOUT = '/proc/1/fd/1';
  const STREAM_CONTAINER_STDERR = '/proc/1/fd/2';

  /**
   * {@inheritdoc}
   */
  public function persist(LoggerEntryInterface $entry): void {
    $value = $entry->__toString();
    if ($this->configuration['max_length'] > 0) {
      $value = LoggerUtils::truncateJsonString($value, $this->configuration['max_length']);
    }
    file_put_contents($this->configuration['stream'], $value . "\n");
  }

  /**
   * {@inheritdoc}
   */
  public function validateTarget(array $target): bool {
    return !empty($target['stream']) && in_array($target['stream'], [
      self::STREAM_STDOUT,
      self::STREAM_STDERR,
      self::STREAM_CONTAINER_STDOUT,
      self::STREAM_CONTAINER_STDERR,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultConfiguration(): array {
    return [
      'destination' => self::STREAM_STDERR,
      'max_length' => 16384,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfigForm() {
    $form['stream'] = [
      // @todo Rework to radios.
      '#type' => 'select',
      '#title' => 'Stream',
      '#default_value' => $this->configuration['stream'],
      '#required' => TRUE,
      '#description' => $this->t('Choose the output stream for the logs. The <code>stderr</code> is recommended to not mix the standard output with logs, for example, in drush commands.'),
      '#options' => [
        self::STREAM_STDERR => $this->t('Standard error output (<code>stderr</code>)'),
        self::STREAM_STDOUT => $this->t('Standard output (<code>stdout</code>)'),
        self::STREAM_CONTAINER_STDOUT => $this->t('Container standard output descriptor (<code>/proc/1/fd/1</code>)'),
        self::STREAM_CONTAINER_STDERR => $this->t('Container error output descriptor (<code>/proc/1/fd/2</code>)'),
      ],
    ];
    $form['max_length'] = [
      '#type' => 'number',
      '#title' => 'Log line maximum length',
      '#default_value' => $this->configuration['max_length'],
      '#required' => FALSE,
      '#min' => 255,
      '#description' => $this->t('The maximum length of a log line. Put a value more than 255, or keep empty to allow any length. Syslog, Docker and some other log receivers pretty often have max line length limits and cut long JSON output producing invalid JSON lines. This usually can be resolved by configuring the receiver configuration. But the module provides a workaround from the Drupal side by cutting the exceeding part of JSON with fixing unclosed brackets and other broken parts. Common limits for Docker: 16384.'),
    ];
    return $form;
  }

}
