<?php

namespace Drupal\logger\Plugin\LoggerTarget;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\logger\Attribute\LoggerTarget;
use Drupal\logger\LoggerEntryInterface;
use Drupal\logger\Plugin\LoggerTargetBase;
use Drupal\logger\Util\LoggerUtils;

/**
 * Provides a syslog logger target.
 */
#[LoggerTarget(
  id: 'syslog',
  label: new TranslatableMarkup('Syslog'),
  description: new TranslatableMarkup('Persists to a syslog daemon. Requires syslog daemon to be available.'),
  weight: 30
)]
class Syslog extends LoggerTargetBase {

  const SYSLOG_ENTRY_MAX_LENGTH_DEFAULT = 2048;

  /**
   * Stores whether there is a system logger connection opened or not.
   *
   * @var bool
   */
  protected $syslogConnectionOpened = FALSE;

  /**
   * {@inheritdoc}
   */
  public function persist(LoggerEntryInterface $entry): void {
    if (!$this->getSyslogConnection($this->configuration['prefix'], $this->configuration['facility'])) {
      throw new \Exception("Can't open the connection to syslog");
    }

    $value = $entry->__toString();
    if ($this->configuration['max_length'] > 0) {
      $value = LoggerUtils::truncateJsonString($value, $this->configuration['max_length']);
    }

    syslog($entry->getLogLevel(), $value);
  }

  /**
   * {@inheritdoc}
   */
  public function validateTarget(array $target): bool {
    return !empty($target['prefix']) && is_numeric($target['facility']);
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultConfiguration(): array {
    return [
      'prefix' => 'drupal',
      // LOG_USER is a common default facility.
      'facility' => LOG_USER,
      'max_length' => self::SYSLOG_ENTRY_MAX_LENGTH_DEFAULT,
    ];
  }

  /**
   * Opens a connection to the system logger.
   *
   * @param string $prefix
   *   The service name prefix.
   * @param int $facility
   *   The syslog facility.
   *
   * @return bool
   *   TRUE if connection was successful, FALSE otherwise.
   */
  protected function getSyslogConnection(string $prefix, int $facility): bool {
    if (!$this->syslogConnectionOpened) {
      $this->syslogConnectionOpened = openlog($prefix, LOG_NDELAY, $facility);
    }
    return $this->syslogConnectionOpened;
  }

  /**
   * {@inheritdoc}
   */
  public function getConfigForm() {
    $form['prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Prefix'),
      '#description' => $this->t('A string that will be prepended to every message logged to Syslog. If you have multiple sites logging to the same Syslog log file, a unique identity per site makes it easy to tell the log entries apart.'),
    ];
    $form['facility'] = [
      '#type' => 'select',
      '#title' => $this->t('Facility'),
      '#options' => $this->syslogFacilityList(),
      '#description' => $this->t('Depending on the system configuration, Syslog and other logging tools use this code to identify or filter messages from within the entire system log.'),
    ];
    $form['max_length'] = [
      '#type' => 'number',
      '#title' => 'Log line maximum length',
      '#default_value' => $this->configuration['max_length'],
      '#required' => FALSE,
      '#min' => 255,
      '#description' => $this->t('The maximum length of a log line. Put a value more than 255, or keep empty to allow any length. Syslog, Docker and some other log receivers pretty often have max line length limits and cut long JSON output producing invalid JSON lines. This usually can be resolved by configuring the receiver configuration. But the module provides a workaround from the Drupal side by cutting the exceeding part of JSON with fixing unclosed brackets and other broken parts. Common limits for Syslog: 2048.'),
    ];
    return $form;
  }

  /**
   * Returns a list of available syslog facilities.
   *
   * @return array
   *   A list with a numeric key and a string value of the each facility.
   */
  protected function syslogFacilityList() {
    return [
      LOG_USER => 'LOG_USER',
      LOG_LOCAL0 => 'LOG_LOCAL0',
      LOG_LOCAL1 => 'LOG_LOCAL1',
      LOG_LOCAL2 => 'LOG_LOCAL2',
      LOG_LOCAL3 => 'LOG_LOCAL3',
      LOG_LOCAL4 => 'LOG_LOCAL4',
      LOG_LOCAL5 => 'LOG_LOCAL5',
      LOG_LOCAL6 => 'LOG_LOCAL6',
      LOG_LOCAL7 => 'LOG_LOCAL7',
    ];
  }

}
