<?php

declare(strict_types=1);

namespace Drupal\Tests\logger\Unit;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\test_helpers\TestHelpers;
use Drupal\Tests\UnitTestCase;
use Drupal\Component\Serialization\Yaml;
use Drupal\logger\Logger\LogMessageParser;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests for the LogMessageParser class.
 */
#[CoversClass(LogMessageParser::class)]
#[Group('logger')]
class LogMessageParserTest extends UnitTestCase {

  use StringTranslationTrait;

  /**
   * Tests placeholder parsing with various formats.
   *
   * Verifies that different types of placeholders (Drupal-style, nested keys,
   * and JSONPath expressions) are properly parsed and replaced.
   */
  public function testParseMessagePlaceholders() {
    TestHelpers::service('string_translation');
    $message = 'Test message with placeholders: @placeholder1, {placeholder2} and {nested.key1}, {nested.key2} and even {$.metadata.values}';
    $context = [
      '@placeholder1' => 'value1',
      'placeholder2' => 'value2',
      'nested' => [
        'key1' => 'nested value 1',
        'key2' => $this->t('translated value 2'),
      ],
      'metadata' => [
        'values' => [
          'val1' => 'My value 1',
          'val2' => 'My value 2',
        ],
      ],
    ];

    $parser = new LogMessageParser();
    $placeholders = $parser->parseMessagePlaceholders($message, $context);
    $messageExpected = $message;
    $expected = [
      '{placeholder2}' => $context['placeholder2'],
      '{nested.key1}' => $context['nested']['key1'],
      '{nested.key2}' => (string) $context['nested']['key2'],
      '{$.metadata.values}' => Yaml::encode($context['metadata']['values']),
      '@placeholder1' => $context['@placeholder1'],
    ];
    $this->assertEquals($messageExpected, $message);
    $this->assertEquals($expected, $placeholders);
  }

  /**
   * Tests handling of invalid placeholder syntax.
   *
   * Verifies that malformed JSONPath expressions are handled gracefully
   * and that error messages are returned for invalid syntax.
   */
  public function testInvalidJsonPlaceholders() {
    $message = 'Test invalid placeholder {metadata.value} {$.metadata:value} {$metadata.value} {metadata.Value} {metadata:value}';
    $context = [
      'metadata' => [
        'value' => 'my_value',
      ],
    ];
    $messageExpected = $message;
    $placeholdersExpected = [
      '{metadata.value}' => "my_value",
      '{$.metadata:value}' => "Unable to parse token metadata:value in expression: .metadata:value",
    ];

    $parser = new LogMessageParser();
    $placeholders = $parser->parseMessagePlaceholders($message, $context);
    $this->assertEquals($messageExpected, $message);
    $this->assertEquals($placeholdersExpected, $placeholders);
  }

}
