<?php

namespace Drupal\logger\Logger;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\DependencyInjection\ContainerInterface;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Logger\LogMessageParserInterface;
use Drupal\Core\Logger\RfcLoggerTrait;
use Drupal\Core\Logger\RfcLogLevel;
use Drupal\logger\Event\LoggerLogEvent;
use Drupal\logger\LoggerEntry;
use Drupal\logger\LoggerEntryInterface;
use Drupal\logger\Plugin\LoggerTargetManager;
use Flow\JSONPath\JSONPath;
use OpenTelemetry\API\Trace\SpanContextInterface;
use OpenTelemetry\API\Trace\SpanInterface;
use OpenTelemetry\SDK\Trace\Span;
use Psr\Log\LoggerInterface;
use Psr\Log\LogLevel;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Redirects logging messages to syslog or stdout.
 */
class Logger implements LoggerInterface {
  use RfcLoggerTrait;

  const CONFIG_NAME = 'logger.settings';

  const CONFIG_KEY_FIELDS = 'fields';
  const CONFIG_KEY_FIELDS_ALL = 'fields_all';
  const CONFIG_KEY_ENTRY_EXCLUDE_EMPTY = 'entry_exclude_empty';
  const CONFIG_KEY_SERVICE_NAME = 'service_name';

  const CONFIG_KEY_TARGETS = 'targets';
  const CONFIG_KEY_EXCEPTION_BACKLOG_ITEMS_LIMIT = 'exception_backlog_items_limit';
  const CONFIG_KEY_SKIP_EVENT_DISPATCH = 'skip_event_dispatch';

  const LOGGER_FIELDS = [
    'service.name' => 'The name of the service that produce the log.',
    'uuid' => 'A universally unique identifier for the log entry.',
    'time' => 'The timestamp as a string implementation in the "c" format.',
    'timestamp' => 'The log entry timestamp.',
    'timestamp_float' => 'The log entry timestamp in milliseconds.',
    'message' => 'The rendered log message with replaced placeholders.',
    'message_raw' => 'The raw log message, without replacing placeholders.',
    'base_url' => 'The base url of the site.',
    'request_time' => 'The main request timestamp.',
    'request_time_float' => 'The main request timestamp in milliseconds.',
    'channel' => 'The log record channel.',
    'ip' => 'The user IP address.',
    'request_uuid' => 'The request UUID. Requires the <code>request_logger</code> module to be installed.',
    'request_uri' => 'The request URI.',
    'referer' => 'The referrer.',
    'severity' => 'The severity level (numeric, 0-7).',
    'level' => 'The severity level in string (error, warning, notice, etc).',
    'uid' => 'The id of the current user.',
    'link' => 'The link value from the log context.',
    'metadata' => 'The structured value of the metadata key in the log context.',
    'exception' => 'Detailed information about an exception.',
    'backtrace' => 'Backtrace array for exceptions. Duplicate the backtrace from the exception field, so do not enable both at once to prevent duplications.',
    'trace_id' => 'OpenTelemetry Trace ID. Requires the OpenTelemetry PHP library.',
    'span_id' => 'OpenTelemetry Span ID, if there is a span in the current context. Requires the OpenTelemetry PHP library.',
  ];

  const JSONPATH_LIBRARY_MISSING_MESSAGE = 'JSONPath library missing';

  /**
   * The logger configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack|null
   */
  protected ?RequestStack $requestStack = NULL;

  /**
   * Constructs a Logger object.
   *
   * @param \Drupal\Component\DependencyInjection\ContainerInterface $container
   *   The service container for lazy loading services.
   * @param \Drupal\Core\Logger\LogMessageParserInterface $parser
   *   The parser to use when extracting message variables.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory.
   * @param \Drupal\logger\Plugin\LoggerTargetManager $pluginManager
   *   The logger target plugin manager.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   */
  public function __construct(
    protected ContainerInterface $container,
    protected LogMessageParserInterface $parser,
    protected ConfigFactoryInterface $configFactory,
    protected LoggerTargetManager $pluginManager,
    protected TimeInterface $time,
  ) {
    $this->config = $this->configFactory->get(self::CONFIG_NAME);
  }

  /**
   * Gets the request stack service.
   */
  protected function getCurrentRequest(): ?Request {
    if ($this->container->has('request_stack') === FALSE) {
      return NULL;
    }
    if ($this->requestStack === NULL) {
      $this->requestStack = $this->container->get('request_stack');
    }
    return $this->requestStack->getCurrentRequest();
  }

  /**
   * Gets the event dispatcher service.
   */
  protected function getEventDispatcher(): EventDispatcherInterface {
    return $this->container->get('event_dispatcher');
  }

  /**
   * {@inheritdoc}
   */
  public function log($level, string|\Stringable $message, array $context = []): void {
    $microtime = $this->time->getCurrentMicroTime();
    $entry = $this->prepareEntry($microtime, $level, $message, $context);

    if (!$this->config->get(self::CONFIG_KEY_SKIP_EVENT_DISPATCH)) {
      $event = new LoggerLogEvent($entry, $level, $message, $context);
      $this->getEventDispatcher()->dispatch($event);
      $entry = $event->entry;
    }

    if ($entry->isEmpty()) {
      return;
    }

    $targets = $this->config->get(self::CONFIG_KEY_TARGETS);
    foreach ($targets as $target) {
      $targetPlugin = $target['plugin'];

      // Check if plugin exists for this target type.
      if (!$this->pluginManager->hasDefinition($targetPlugin)) {
        throw new \Exception("Configured log target \"{$targetPlugin}\" is not supported.");
      }

      $configuration = json_decode($target['configuration'], associative: TRUE);
      if ($level > $target['log_level']) {
        continue;
      }

      try {
        $plugin = $this->pluginManager->createInstance($targetPlugin, $configuration);
        $plugin->persist($entry, $level);
      }
      catch (\Exception $e) {
        throw new \Exception("Error persisting to target \"{$targetPlugin}\": " . $e->getMessage(), 0, $e);
      }
    }
  }

  /**
   * Returns a list of enabled fields in the configuration.
   */
  public function getEnabledFields(): array {
    return $this->config->get(self::CONFIG_KEY_FIELDS) ?? [];
  }

  /**
   * Prepares a log entry object from the message and context.
   *
   * @param float $microtime
   *   The microtime of the entry creation.
   * @param int $level
   *   The log level.
   * @param string $message
   *   The log message or message template.
   * @param array $context
   *   The log context array.
   *
   * @return \Drupal\logger\LoggerEntryInterface
   *   The prepared log entry instance.
   */
  protected function prepareEntry(float $microtime, int $level, string $message, array $context): LoggerEntryInterface {
    $fieldsEnabled = $this->config->get(self::CONFIG_KEY_FIELDS) ?? [];

    if (
      isset($context['backtrace'])
      && $limit = $this->config->get(self::CONFIG_KEY_EXCEPTION_BACKLOG_ITEMS_LIMIT)
    ) {
      $context['backtrace'] = array_slice($context['backtrace'], 0, $limit);
    }

    $entry = new LoggerEntry(
      logLevel: $level,
      data: [],
      microtime: $microtime,
    );
    foreach ($fieldsEnabled as $field) {
      switch ($field) {
        case 'service.name':
          if ($value = $this->config->get(self::CONFIG_KEY_SERVICE_NAME)) {
            $entry->set($field, $value);
          }
          break;

        case 'message':
          $messageCopy = $message;
          $messagePlaceholders ??= $this->parser->parseMessagePlaceholders($messageCopy, $context);
          $messageRendered = empty($messagePlaceholders) ? $message : strtr($messageCopy, $messagePlaceholders);
          $entry->set($field, $messageRendered);
          break;

        case 'message_raw':
          $messageCopy = $message;
          $messagePlaceholders ??= $this->parser->parseMessagePlaceholders($messageCopy, $context);
          $entry->set($field, $messageCopy);
          $entryData = $entry->getData();
          foreach ($messagePlaceholders as $key => $value) {
            $valuePath = $this->getPlaceholderPath($key);
            NestedArray::setValue($entryData, $valuePath, $value);
          }
          $entry->setData($entryData);
          break;

        case 'base_url':
          global $base_url;
          $entry->set($field, $base_url);
          break;

        case 'timestamp_float':
          // We have no microsecond information from the context, therefore
          // have to log the current time with microseconds.
          $entry->set($field, $entry->getMicrotime());
          break;

        case 'time':
          $mt = $entry->getMicrotime();
          $microtimeString = date('Y-m-d\TH:i:s', floor($mt)) . sprintf('.%06d', ($mt - floor($mt)) * 1000000) . date('P', floor($mt));
          $entry->set($field, $microtimeString);
          break;

        case 'uuid':
          $entry->set($field, $this->container->get('uuid')->generate());
          break;

        case 'request_uuid':
          if (
            ($request ??= $this->getCurrentRequest())
            && $value = $request->attributes->get('request_uuid')
          ) {
            $entry->set($field, $value);
          }
          break;

        case 'request_time':
          if ($request ??= $this->getCurrentRequest()) {
            $entry->set($field, $request->server->get('REQUEST_TIME'));
          }
          break;

        case 'request_time_float':
          if ($request ??= $this->getCurrentRequest()) {
            $entry->set($field, $request->server->get('REQUEST_TIME_FLOAT'));
          }
          break;

        case 'severity':
          $entry->set($field, $level);
          break;

        case 'level':
          $entry->set($field, self::getRfcLogLevelAsString($level));
          break;

        case 'exception':
          if (array_key_exists($field, $context)) {
            if ($context['exception'] instanceof \Throwable) {
              // We use a custom implementation instead of the
              // Drupal\Core\Utility\Error::decodeException()
              // to produce the array in a more standard way.
              $entry->set($field, $this->exceptionToArray($context['exception']));
            }
            else {
              $entry->set($field, $context['exception']);
            }
          }
          break;

        // A special label "metadata" to pass any free form data.
        case 'metadata':
          if (array_key_exists($field, $context)) {
            $entry->set($field, $context[$field]);
          }
          break;

        // Default context keys from Drupal Core.
        case 'timestamp':
        case 'channel':
        case 'ip':
        case 'request_uri':
        case 'referer':
        case 'uid':
        case 'link':
        case 'backtrace':
        default:
          if (array_key_exists($field, $context)) {
            $entry->set($field, $context[$field]);
          }
          break;
      }
    }

    if ($this->config->get(self::CONFIG_KEY_FIELDS_ALL) ?? FALSE) {
      foreach ($context as $field => $value) {
        if (!isset($fieldsEnabled[$field])) {
          $entry->set($field, $value);
        }
      }
    }

    // If we have an OpenTelemetry span, add the trace id to the log entry.
    if (
      (in_array('trace_id', $fieldsEnabled) && in_array('span_id', $fieldsEnabled))
      && class_exists(Span::class)
    ) {
      $span = Span::getCurrent();
      if ($span instanceof SpanInterface) {
        $spanContext = $span->getContext();
        if (
          $spanContext instanceof SpanContextInterface
          && $spanContext->isValid()
        ) {
          $traceId = $spanContext->getTraceId();
          if (in_array('trace_id', $fieldsEnabled)) {
            $entry->set('trace_id', $traceId);
          }
          if (in_array('span_id', $fieldsEnabled)) {
            $entry->set('span_id', $spanContext->getSpanId());
          }
        }
      }
    }

    if ($this->config->get(self::CONFIG_KEY_ENTRY_EXCLUDE_EMPTY) ?? FALSE) {
      $entry->cleanEmptyValues();
    }

    return $entry;
  }

  /**
   * Builds an array path from a placeholder string.
   *
   * Supports Drupal placeholders, PSR-3 style placeholders and simple
   * JSONPath expressions starting with ``$.``.
   *
   * @param string $placeholder
   *   The placeholder string (for example ``{$.foo.bar}`` or ``{foo.bar}``).
   *
   * @return array
   *   The path as an array of parts.
   */
  private function getPlaceholderPath(string $placeholder): array {
    if (str_starts_with($placeholder, '{') && str_ends_with($placeholder, '}')) {
      $path = substr($placeholder, 1, -1);
      if (str_starts_with($path, '$.')) {
        $path = substr($path, 2);
      }
      return explode('.', $path);
    }
    return [$placeholder];
  }

  /**
   * Gets a value from a nested data array by a placeholder expression.
   *
   * Supports Drupal placeholders, PSR-3 and JSONPath expressions. Returns
   * the found value or NULL when not present.
   *
   * @param array $data
   *   The data array to search.
   * @param string $placeholder
   *   The placeholder expression.
   *
   * @return mixed
   *   The found value or NULL if not found.
   */
  private function getValueByPlaceholder(array $data, string $placeholder) {
    if (str_starts_with($placeholder, '{') && str_ends_with($placeholder, '}')) {
      $expression = substr($placeholder, 1, -1);
      if (str_starts_with($expression, '$.')) {
        return self::getJsonPathValue($data, $expression);
      }
      else {
        $parts = explode('.', $expression);
        return NestedArray::getValue($data, $parts);
      }
    }
    return $data[$placeholder] ?? NULL;
  }

  /**
   * Retrieves a value from an array using a JSONPath expression.
   *
   * If the optional JSONPath library is not available, this method returns
   * a special message constant.
   *
   * @param array $data
   *   The array to search.
   * @param string $field
   *   The JSONPath expression to use (for example ``$.foo.bar``).
   *
   * @return mixed
   *   The value found at the specified JSONPath, a string error message if
   *   the JSONPath library is missing, or NULL if not found.
   */
  public static function getJsonPathValue(array $data, string $field) {
    if (!class_exists(JSONPath::class)) {
      return self::JSONPATH_LIBRARY_MISSING_MESSAGE;
    }
    $jsonData = new JSONPath($data);
    try {
      $value = $jsonData->find($field)->getData();
    }
    catch (\Exception $e) {
      $value = [$e->getMessage()];
    }

    // Empty array means that the value was not found.
    if (is_array($value) && empty($value)) {
      return NULL;
    }

    if (is_array($value) && count($value) <= 1) {
      // If the JSONPath returns a single value, we can use it directly.
      $value = reset($value);
    }

    return $value;
  }

  /**
   * Convert a level integer to a string representation of the RFC log level.
   *
   * @param int $level
   *      The log message level.
   *
   * @return string
   *      String representation of the log level.
   */
  public static function getRfcLogLevelAsString(int $level): string {
    return match ($level) {
      RfcLogLevel::EMERGENCY => LogLevel::EMERGENCY,
      RfcLogLevel::ALERT => LogLevel::ALERT,
      RfcLogLevel::CRITICAL => LogLevel::CRITICAL,
      RfcLogLevel::ERROR => LogLevel::ERROR,
      RfcLogLevel::WARNING => LogLevel::WARNING,
      RfcLogLevel::NOTICE => LogLevel::NOTICE,
      RfcLogLevel::INFO => LogLevel::INFO,
      RfcLogLevel::DEBUG => LogLevel::DEBUG,
    };
  }

  /**
   * Converts an exception to the associative array representation.
   *
   * @param \Throwable $e
   *   An exception.
   *
   * @return array
   *   An associative array with the exception data.
   */
  private function exceptionToArray(\Throwable $e) {
    $array = [
      'message' => $e->getMessage(),
      'code' => $e->getCode(),
      'file' => $e->getFile(),
      'line' => $e->getLine(),
      'trace' => $e->getTrace(),
    ];

    if ($limit = $this->config->get(self::CONFIG_KEY_EXCEPTION_BACKLOG_ITEMS_LIMIT)) {
      $array['trace'] = array_slice($array['trace'], 0, $limit);
    }

    if ($ePrevious = $e->getPrevious()) {
      $array['previous'] = $this->exceptionToArray($ePrevious);
    }
    return $array;
  }

}
