<?php

namespace Drupal\logger;

/**
 * The interface for a Logger entry class.
 *
 * The class contains all the data that will be logged.
 *
 * @see Drupal\logger\Logger\Logger::LOGGER_FIELDS for the list
 * of common possible values.
 */
interface LoggerEntryInterface {

  /**
   * The LoggerEntry constructor.
   *
   * @param int $logLevel
   *   The RFC log level.
   * @param array $data
   *   (optional) An array with initial data.
   * @param float|null $microtime
   *   (optional) A float with the microtime when the log entry was created.
   *   If not provided, the current microtime will be used.
   */
  public function __construct(
    int $logLevel,
    array $data = [],
    ?float $microtime = NULL,
  );

  /**
   * Sets a value to the log entry by a key.
   *
   * @param string $key
   *   A key.
   * @param mixed $value
   *   A value.
   *
   * @return \Drupal\logger\LoggerEntry
   *   The LoggerEntry object.
   */
  public function set(string $key, $value): LoggerEntry;

  /**
   * Gets a value from the log entry by a key.
   *
   * @param string $key
   *   A key.
   *
   * @return mixed
   *   The value by the key.
   */
  public function get(string $key): mixed;

  /**
   * Gets the log level of the entry.
   */
  public function getLogLevel(): int;

  /**
   * Gets the microtime of the entry.
   */
  public function getMicrotime(): float;

  /**
   * Deletes a value from the log entry by a key.
   *
   * @param string $key
   *   A key.
   *
   * @return \Drupal\logger\LoggerEntry
   *   The LoggerEntry object.
   */
  public function delete(string $key): LoggerEntry;

  /**
   * Sets the whole data of the log entry.
   *
   * @param array $data
   *   The array with data.
   *
   * @return \Drupal\logger\LoggerEntry
   *   The LoggerEntry object.
   */
  public function setData(array $data): LoggerEntry;

  /**
   * Gets the whole log entry data.
   */
  public function getData(): mixed;

  /**
   * Cleans up empty values from the log entry data, but keeping numeric zeros.
   */
  public function cleanEmptyValues(): void;

  /**
   * Checks if the log entry is empty.
   */
  public function isEmpty(): bool;

  /**
   * Converts the log data to the string representation.
   */
  public function __toString(): string;

}
