<?php

declare(strict_types=1);

namespace Drupal\Tests\logger\Unit;

use Drupal\Core\Logger\RfcLogLevel;
use Drupal\logger\LoggerEntry;
use Drupal\logger\Util\LoggerUtils;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests for the LoggerEntry class.
 */
#[CoversClass(LoggerEntry::class)]
#[Group('logger')]
class LoggerEntryTest extends UnitTestCase {

  /**
   * Tests general LoggerEntry functionality.
   *
   * This test covers basic operations like construction, getting/setting data,
   * deleting fields, and string conversion.
   */
  public function testGeneral() {
    $entry0 = new LoggerEntry(RfcLogLevel::DEBUG);
    $this->assertEquals([], $entry0->getData());

    $data1 = ['foo' => 'bar', 'baz' => 'qix'];
    $entry1 = new LoggerEntry(RfcLogLevel::EMERGENCY, $data1);
    $this->assertEquals($data1, $entry1->getData());
    $this->assertEquals($data1['foo'], $entry1->get('foo'));

    $data2 = $data1 + ['fred' => 'thud'];
    $entry1->set('fred', $data2['fred']);
    $this->assertEquals($data2, $entry1->getData());
    $this->assertEquals($data2['fred'], $entry1->get('fred'));
    $this->assertEquals(json_encode($data2), $entry1->__toString());

    $entry1->delete('fred');
    $this->assertEquals($data1, $entry1->getData());

    $entry1->setData($data2);
    $this->assertEquals($data2['fred'], $entry1->get('fred'));
  }

  /**
   * Tests entry string conversion and truncation.
   *
   * Verifies that log entries are properly converted to JSON strings
   * and truncated when they exceed configured length limits.
   */
  public function testGetEntryAsString() {
    $entryData = [
      'service.name' => 'drupal',
      'timestamp_float' => 12345.984621,
      'message' => 'My message',
      'metadata' => [
        'foo' => [
          'bar' => 'baz',
        ],
        'qix' => [
          12.3,
          45.6,
        ],
      ],
    ];
    $entryAsJson = json_encode($entryData);
    $entryAsJsonLength = strlen($entryAsJson);

    // Test without truncation (normal case).
    $entry = new LoggerEntry(RfcLogLevel::DEBUG, $entryData);
    $this->assertEquals($entryAsJson, LoggerUtils::truncateJsonString($entry->__toString(), $entryAsJsonLength));

    // Dangerous cut points that can lead to invalid JSON.
    $dangerousCutPoints = [
      -1,
      -18,
    ];

    foreach ($dangerousCutPoints as $cutPoint) {
      $entry = new LoggerEntry(
        RfcLogLevel::DEBUG,
        $entryData,
      );
      $result = LoggerUtils::truncateJsonString($entry->__toString(), $entryAsJsonLength + $cutPoint);
      $this->assertJson($result);
      $this->assertLessThanOrEqual($entryAsJsonLength, strlen($result));
      // The result should contain the cut indicator if truncated.
      $this->assertStringContainsString(LoggerUtils::CUT_SUFFIX, $result);
    }
  }

}
