<?php

declare(strict_types=1);

namespace Drupal\Tests\logger\Unit;

use Drupal\Core\Logger\RfcLogLevel;
use Drupal\Core\StreamWrapper\StreamWrapperInterface;
use Drupal\Core\Utility\Error;
use Drupal\logger\Logger\Logger;
use Drupal\logger\Logger\LogMessageParser;
use Drupal\logger\Plugin\LoggerTarget\File;
use Drupal\test_helpers\TestHelpers;
use Drupal\Tests\logger\Utils\LogTargetTest1Plugin;
use Drupal\Tests\logger\Utils\LogTargetTest2Plugin;
use Drupal\Tests\logger\Utils\UtilsTrait;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use Psr\Log\LogLevel;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Yaml\Yaml;

/**
 * Unit tests for the Logger service.
 */
#[CoversClass(Logger::class)]
#[Group('logger')]
class LoggerTest extends UnitTestCase {

  use UtilsTrait;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();
    // Ensure that the static cache is cleared before each test.
    LogTargetTest1Plugin::resetStaticCache();
    LogTargetTest2Plugin::resetStaticCache();
  }

  /**
   * Test the main logging workflow including entry preparation and persistence.
   */
  public function testLog() {
    global $base_url;
    $base_url = 'https://example.com/path';
    $server['REQUEST_TIME'] = 123;
    $server['REQUEST_TIME_FLOAT'] = 123.456;
    $request = new Request(server: $server);
    TestHelpers::service(LogMessageParser::class, new LogMessageParser());
    TestHelpers::service('request_stack')->push($request);

    $config = Yaml::parseFile(TestHelpers::getModuleFilePath('config/install/logger.settings.yml'));

    $plugin1Configuration = ['foo' => 'baz'];
    $plugin2Configuration = ['foo' => 'qux'];
    $pluginManager = TestHelpers::service('plugin.manager.logger_target', initService: TRUE);
    $pluginDefinitions = [
      'test1' => [
        'class' => LogTargetTest1Plugin::class,
        'provider' => 'logger',
      ],
      'test2' => [
        'class' => LogTargetTest2Plugin::class,
        'provider' => 'logger',
      ],
    ];
    TestHelpers::setPrivateProperty($pluginManager, 'definitions', $pluginDefinitions);
    $config['targets'] = [
      [
        'plugin' => 'test1',
        'log_level' => RfcLogLevel::INFO,
        'configuration' => json_encode($plugin1Configuration),
      ],
      [
        'plugin' => 'test2',
        'log_level' => RfcLogLevel::ERROR,
        'configuration' => json_encode($plugin2Configuration),
      ],
    ];

    $config['fields'] = array_merge($config['fields'], [
      'customField2',
      'custom_field_5',
      'custom_field_6',
      'custom_field_7',
      'custom_field_8',
    ]);
    $config['entry_exclude_empty'] = TRUE;
    $config['service.name'] = 'My cool service name';
    TestHelpers::service('config.factory')->stubSetConfig(Logger::CONFIG_NAME, $config);

    $context = [
      'ip' => '192.168.1.1',
      'timestamp' => 1234567,
      'customField2' => 0,
      'customField3' => 'custom3 value',
      'custom_field_6' => NULL,
      'custom_field_8' => '0',
      'metadata' => ['foo' => ['bar' => 'baz']],
      '@my_placeholder' => 'Bob',
    ];
    $message_raw = 'A message from @my_placeholder!';
    $message = "A message from {$context['@my_placeholder']}!";

    $resultEntryValuesAll = [
      // The 'timestamp_float' is not static, checked separately.
      'message' => $message,
      'message_raw' => $message_raw,
      '@my_placeholder' => 'Bob',
      'base_url' => $base_url,
      'request_time_float' => $server['REQUEST_TIME_FLOAT'],
      'ip' => '192.168.1.1',
      'severity' => 4,
      'level' => 'warning',
      'metadata' => $context['metadata'],
      'customField2' => $context['customField2'],
      'custom_field_6' => $context['custom_field_6'],
      'custom_field_8' => '0',
      // Custom values should be at the end of the entry.
      'customField3' => $context['customField3'],
    ];

    $logger1 = TestHelpers::initService('logger');

    $logger1->log(RfcLogLevel::INFO, $message_raw, $context);
    $resultEntryExpected1 = $resultEntryValuesAll;
    $resultEntryExpected1['severity'] = RfcLogLevel::INFO;
    $resultEntryExpected1['level'] = LogLevel::INFO;
    unset($resultEntryExpected1['custom_field_6']);
    unset($resultEntryExpected1['customField3']);

    $logger1->log(RfcLogLevel::DEBUG, $message_raw, $context);

    // Test "fields_all".
    $config['fields_all'] = TRUE;
    $config['entry_exclude_empty'] = FALSE;
    TestHelpers::service('config.factory')->stubSetConfig(Logger::CONFIG_NAME, $config);

    $logger2 = TestHelpers::initService('logger');
    $logger2->log(RfcLogLevel::ERROR, $message_raw, $context);
    $resultEntryExpected3 = $resultEntryValuesAll;
    $resultEntryExpected3['severity'] = RfcLogLevel::ERROR;
    $resultEntryExpected3['level'] = LogLevel::ERROR;

    // Create a new plugin instance to check the static $logs property.
    $plugin1 = $pluginManager->createInstance('test1');
    $this->assertCount(2, $plugin1::$logs);
    $this->assertEquals($plugin1Configuration, $plugin1::$logs[0]['configuration']);
    $this->assertEquals(json_encode($resultEntryExpected1), $this->checkAndRemoveDynamicEntryValues($plugin1::$logs[0]['entry']->__toString()));
    $this->assertEquals(json_encode($resultEntryExpected3), $this->checkAndRemoveDynamicEntryValues($plugin1::$logs[1]['entry']->__toString()));

    $plugin2 = $pluginManager->createInstance('test2');
    $this->assertCount(1, $plugin2::$logs);
    $this->assertEquals($plugin2Configuration, $plugin2::$logs[0]['configuration']);
    $this->assertEquals(json_encode($resultEntryExpected3), $this->checkAndRemoveDynamicEntryValues($plugin2::$logs[0]['entry']->__toString()));
    TestHelpers::service('request_stack')->pop();
  }

  /**
   * Tests that logger doesn't throw errors with an empty configuration.
   */
  public function testLogWithEmptyConfig() {
    TestHelpers::service(LogMessageParser::class, new LogMessageParser());
    TestHelpers::service('logger', initService: TRUE);
    \Drupal::service('logger')->log(RfcLogLevel::DEBUG, 'Test message1');
    \Drupal::service('logger')->log(RfcLogLevel::WARNING, 'Test message2');
    $this->assertEmpty(LogTargetTest1Plugin::$logs);
  }

  /**
   * Tests that logger works well with default config.
   */
  public function testLogWithDefaultConfig() {
    $pluginManager = TestHelpers::service('plugin.manager.logger_target', initService: TRUE);
    $pluginDefinitions = [
      'file' => [
        'class' => File::class,
        'provider' => 'logger',
      ],
    ];
    TestHelpers::setPrivateProperty($pluginManager, 'definitions', $pluginDefinitions);

    $config = Yaml::parseFile(TestHelpers::getModuleFilePath('config/install/logger.settings.yml'));
    $target1Configuration = json_decode($config['targets'][0]['configuration'] ?? '{}', associative: TRUE);
    $filePath = str_replace('temporary:/', sys_get_temp_dir(), $target1Configuration['destination'] ?? '');
    if (file_exists($filePath)) {
      unlink($filePath);
    }
    $streamWrapperManager = TestHelpers::service('stream_wrapper_manager', initService: FALSE);
    $streamWrapperInterface = $this->createMock(StreamWrapperInterface::class);
    $streamWrapperInterface->method('realpath')->willReturn($filePath);
    $streamWrapperManager->method('getViaUri')->willReturn($streamWrapperInterface);

    TestHelpers::service('config.factory')->stubSetConfig(Logger::CONFIG_NAME, $config);
    TestHelpers::service(LogMessageParser::class, new LogMessageParser());

    TestHelpers::service('logger', initService: TRUE);
    global $base_url;
    $base_url = 'https://example.com/path2';
    \Drupal::service('logger')->log(RfcLogLevel::DEBUG, 'Test message 1');
    $base_url = 'https://example.com/path3';
    \Drupal::service('logger')->log(RfcLogLevel::WARNING, 'Test message 2');
    $this->assertEmpty(LogTargetTest1Plugin::$logs);

    $logs = file($filePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    $this->assertCount(2, $logs);
    $entry1 = $this->checkAndRemoveDynamicEntryValues($logs[0], logLevel: RfcLogLevel::DEBUG);
    $entry1Array = json_decode($entry1, TRUE);
    $this->assertIsFloat($entry1Array['request_time_float']);
    unset($entry1Array['request_time_float']);
    $entry1 = json_encode($entry1Array);
    $entry1Expected = '{"message":"Test message 1","message_raw":"Test message 1","base_url":"https:\/\/example.com\/path2"}';
    $this->assertEquals($entry1Expected, $entry1);

    $entry2 = $this->checkAndRemoveDynamicEntryValues($logs[1], logLevel: RfcLogLevel::WARNING);
    $entry2Array = json_decode($entry2, TRUE);
    $this->assertIsFloat($entry2Array['request_time_float']);
    unset($entry2Array['request_time_float']);
    $entry2 = json_encode($entry2Array);
    $entry2Expected = '{"message":"Test message 2","message_raw":"Test message 2","base_url":"https:\/\/example.com\/path3"}';
    $this->assertEquals($entry2Expected, $entry2);
  }

  /**
   * Tests logging entries containing exception data.
   *
   * Verifies that exceptions are properly converted to arrays and that
   * backtrace items are limited according to configuration.
   */
  public function testEntryWithException() {
    $limit = 3;
    $config = [
      'fields' => [
        'exception',
        'backtrace',
      ],
      'exception_backlog_items_limit' => $limit,
    ];
    TestHelpers::service('config.factory')->stubSetConfig(Logger::CONFIG_NAME, $config);

    TestHelpers::service(LogMessageParser::class, new LogMessageParser());
    $logger = TestHelpers::initService('logger');
    $exception = new \Exception('Test exception', 0, new \Exception('Inner exception'));
    $context = Error::decodeException($exception);

    $entry = TestHelpers::callPrivateMethod($logger, 'prepareEntry', [
      microtime(TRUE),
      Error::ERROR,
      Error::DEFAULT_ERROR_MESSAGE,
      $context,
    ]);

    $log = $entry->getData();
    $this->assertCount($limit, $log['exception']['trace']);
    $this->assertCount($limit, $log['backtrace']);
  }

  /**
   * Tests placeholder replacement using JSONPath notation.
   *
   * Verifies that placeholders using dot notation and JSONPath syntax
   * are properly resolved from context data in log messages.
   */
  public function testEntryWithJsonPathPlaceholders() {
    $config = [
      'fields' => [
        'message',
        'message_raw',
        'customField2',
      ],
    ];
    $message = 'Test: {$.customField2.foo.bar} {customField1.subField2} prefix_@placeholder2_suffix. End';
    $context = [
      'customField1' => [
        'subField1' => 'subValue1',
        'subField2' => 'subValue2',
      ],
      '@placeholder1' => 'placeholderValue1',
      '@placeholder2' => 'placeholderValue2',
      'customField2' => [
        'foo' => [
          'bar' => 'baz',
          'qix' => 'qux',
        ],
      ],
    ];

    TestHelpers::service('config.factory')->stubSetConfig(Logger::CONFIG_NAME, $config);
    TestHelpers::service(LogMessageParser::class, new LogMessageParser());
    $logger = TestHelpers::initService('logger');
    $entry = TestHelpers::callPrivateMethod($logger, 'prepareEntry', [
      microtime(TRUE),
      RfcLogLevel::INFO,
      $message,
      $context,
    ]);
    $log = $entry->getData();

    $this->assertEquals($message, $log['message_raw']);
    $this->assertEquals('Test: baz subValue2 prefix_placeholderValue2_suffix. End', $log['message']);

    $this->assertEquals($context['customField1']['subField2'], $log['customField1']['subField2']);
    // This value should be absent, because the customField1 is not selected to
    // store in the config, and no usage of this value in the placeholders.
    $this->assertFalse(isset($log['customField1']['subField1']));

    $this->assertEquals($context['customField2']['foo']['bar'], $log['customField2']['foo']['bar']);
    // This value should be present, because the customField2 is selected to
    // store in the config.
    $this->assertEquals($context['customField2']['foo']['qix'], $log['customField2']['foo']['qix']);

    $this->assertEquals($context['@placeholder1'], $log['@placeholder1']);
    // This value should be present, because Drupal Core's function
    // parseMessagePlaceholders() extracts all Drupal-related placeholders.
    $this->assertEquals($context['@placeholder2'], $log['@placeholder2']);

    $config = [
      'fields' => [
        'message',
        'message_raw',
        'customField2',
      ],
      'fields_all' => TRUE,
    ];
    TestHelpers::service('config.factory')->stubSetConfig(Logger::CONFIG_NAME, $config);
    TestHelpers::service(LogMessageParser::class, new LogMessageParser());
    $logger = TestHelpers::initService('logger');
    $entry = TestHelpers::callPrivateMethod($logger, 'prepareEntry', [
      microtime(TRUE),
      RfcLogLevel::INFO,
      $message,
      $context,
    ]);
    $log = $entry->getData();
    $this->assertTrue(TestHelpers::isNestedArraySubsetOf($log, $context));
  }

  /**
   * Tests JSONPath value extraction functionality.
   *
   * Verifies that values can be extracted from nested arrays using both
   * simple dot notation and JSONPath expressions, including error handling.
   */
  public function testGetJsonPathValue() {
    $data = [
      'foo' => [
        'bar' => 'baz',
        'qix' => 'qux',
      ],
    ];
    // Test simple dot notation.
    $simplePath = 'foo.bar';
    $result = Logger::getJsonPathValue($data, $simplePath);
    $this->assertEquals('baz', $result);

    // Test JSONPath notation.
    $jsonPath = '$.foo.bar';
    $resultJsonPath = Logger::getJsonPathValue($data, $jsonPath);
    if ($resultJsonPath === Logger::JSONPATH_LIBRARY_MISSING_MESSAGE) {
      $this->markTestSkipped('JSONPath library missing');
    }
    else {
      $this->assertEquals('baz', $resultJsonPath);
    }

    // Test JSONPath exception.
    $data = [
      'foo' => [
        'bar' => 'baz',
      ],
    ];
    $invalidJsonPath = '{$.foo[INVALID]}';
    $result = Logger::getJsonPathValue($data, $invalidJsonPath);
    // The result should be a string with the error message.
    $this->assertIsString($result);
    $this->assertStringContainsString('Unable to parse token', $result);
  }

  /**
   * Checks and removes dynamic values from a log entry string.
   */
  private function checkAndRemoveDynamicEntryValues(string $entry, ?int $logLevel = NULL, string $serviceName = 'drupal'): string {
    $entryArray = json_decode($entry, associative: TRUE);
    $this->assertEquals($serviceName, $entryArray['service.name']);
    unset($entryArray['service.name']);

    if ($logLevel !== NULL) {
      $this->assertEquals($logLevel, $entryArray['severity']);
      $this->assertEquals(Logger::getRfcLogLevelAsString($logLevel), $entryArray['level']);
      unset($entryArray['level']);
      unset($entryArray['severity']);
    }

    if (isset($entryArray['time'])) {
      $this->assertIsString($entryArray['time']);
      unset($entryArray['time']);
    }
    if (isset($entryArray['timestamp_float'])) {
      $this->assertIsFloat($entryArray['timestamp_float']);
      unset($entryArray['timestamp_float']);
    }
    if (isset($entryArray['timestamp'])) {
      $this->assertIsInt($entryArray['timestamp']);
      unset($entryArray['timestamp']);
    }
    return json_encode($entryArray);
  }

}
