<?php

namespace Drupal\logger_db\Form;

use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\logger\Logger\Logger;
use Drupal\logger\Trait\SettingLabelTrait;
use Drupal\logger_db\LoggerDbManager;
use Drupal\logger_db\LoggerDbUtils;
use Drupal\Core\Url;

use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Logger DB settings for this site.
 */
class SettingsForm extends ConfigFormBase {
  use SettingLabelTrait;

  /**
   * A TypedConfigManager.
   *
   * @var \Drupal\Core\Config\TypedConfigManagerInterface
   */
  protected TypedConfigManagerInterface $configTyped;

  /**
   * An LoggerDbManager.
   *
   * @var \Drupal\logger_db\LoggerDbManager
   */
  protected LoggerDbManager $loggerDbManager;

  /**
   * An LoggerDbUtils.
   *
   * @var \Drupal\logger_db\LoggerDbUtils
   */
  protected LoggerDbUtils $loggerDbUtils;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->configTyped = $container->get('config.typed');
    $instance->loggerDbManager = $container->get(LoggerDbManager::class);
    $instance->loggerDbUtils = $container->get(LoggerDbUtils::class);
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'logger_db_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [LoggerDbManager::CONFIG_KEY];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $this->settingsTyped = $this->configTyped->get(LoggerDbManager::CONFIG_KEY);

    $loggerSettings = $this->config(Logger::CONFIG_NAME);
    $targets = $loggerSettings->get(Logger::CONFIG_KEY_TARGETS);
    $pluginEnabled = FALSE;
    foreach ($targets as $target) {
      if ($target['plugin'] == 'db') {
        $pluginEnabled = TRUE;
      }
    }
    if (!$pluginEnabled) {
      $this->messenger()->addError($this->t('The Database Logger plugin is not enabled. Open the <a href="@url">Logger Settings page</a> and choose the "Database" plugin in the target.', [
        '@url' => Url::fromRoute('logger.settings')->toString(),
      ]));
    }

    $form['logs_view_page'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Logs View Page'),
    ];

    $form['logs_view_page']['description'] = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#value' => $this->t('<a href="/admin/reports/logs">Open the Logs View Page</a>. You can customize the columns and other settings on the views edit page <a href="/admin/structure/views/view/logger_db_logs/edit">here</a>.'),
    ];
    $form['logs_view_page']['reset'] = [
      '#type' => 'details',
      '#title' => $this->t('Reset the Logs page configuration to defaults'),
      '#description' => $this->t('Use the button below to reset your changes back to defaults. Warning: it will remove all customizations of the log page and reset the page to the default settings.'),
      '#open' => FALSE,
    ];
    $form['logs_view_page']['reset']['reset_logs_view_page'] = [
      '#type' => 'submit',
      '#name' => 'reset_logs_view_page',
      '#value' => $this->t('Reset Logs View Page settings to defaults'),
      '#attributes' => [
        'class' => ['button--danger'],
      ],
    ];

    $form['cleanup'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Cleanup rules'),
      '#description' => $this->t('Old log entries cleans up automatically by cron. Here you can configure cleanup rules.'),
      '#description_display' => 'before',
    ];
    $form['cleanup'][LoggerDbManager::CONFIG_KEY_CLEANUP_BY_TIME_ENABLED] = [
      '#type' => 'checkbox',
      '#title' => $this->getSettingLabel(LoggerDbManager::CONFIG_KEY_CLEANUP_BY_TIME_ENABLED),
      '#description' => $this->t('Enables deleting old log records by time.'),
      '#config_target' => LoggerDbManager::CONFIG_KEY . ':' . LoggerDbManager::CONFIG_KEY_CLEANUP_BY_TIME_ENABLED,
    ];

    $form['cleanup'][LoggerDbManager::CONFIG_KEY_CLEANUP_BY_TIME_SECONDS] = [
      '#type' => 'select',
      '#title' => $this->getSettingLabel(LoggerDbManager::CONFIG_KEY_CLEANUP_BY_TIME_SECONDS),
      '#description' => $this->t('Time range to store.'),
      '#options' => [
        60 * 60 * 24 * 7 => $this->t('1 week'),
        60 * 60 * 24 * 14 => $this->t('2 weeks'),
        60 * 60 * 24 * 28 => $this->t('4 weeks'),
        60 * 60 * 24 * 31 => $this->t('1 month'),
        60 * 60 * 24 * 31 * 3 => $this->t('3 months'),
        60 * 60 * 24 * 31 * 6 => $this->t('6 months'),
        60 * 60 * 24 * 365 => $this->t('1 year'),
      ],
      '#config_target' => LoggerDbManager::CONFIG_KEY . ':' . LoggerDbManager::CONFIG_KEY_CLEANUP_BY_TIME_SECONDS,
      '#states' => [
        'visible' => [
          ':input[name="' . LoggerDbManager::CONFIG_KEY_CLEANUP_BY_TIME_ENABLED . '"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['cleanup'][LoggerDbManager::CONFIG_KEY_CLEANUP_BY_ROWS_ENABLED] = [
      '#type' => 'checkbox',
      '#title' => $this->getSettingLabel(LoggerDbManager::CONFIG_KEY_CLEANUP_BY_ROWS_ENABLED),
      '#description' => $this->t('Enables deleting old log records by the total amount of records.'),
      '#config_target' => LoggerDbManager::CONFIG_KEY . ':' . LoggerDbManager::CONFIG_KEY_CLEANUP_BY_ROWS_ENABLED,
    ];

    $form['cleanup'][LoggerDbManager::CONFIG_KEY_CLEANUP_BY_ROWS_LIMIT] = [
      '#type' => 'select',
      '#title' => $this->getSettingLabel(LoggerDbManager::CONFIG_KEY_CLEANUP_BY_ROWS_LIMIT),
      '#description' => $this->t('Amount of records to store.'),
      '#options' => [
        1_000 => $this->t('@count rows', ['@count' => '1 000']),
        10_000 => $this->t('@count rows', ['@count' => '10 000']),
        100_000 => $this->t('@count rows', ['@count' => '100 000']),
        1_000_000 => $this->t('@count rows', ['@count' => '1 000 000']),
      ],
      '#config_target' => LoggerDbManager::CONFIG_KEY . ':' . LoggerDbManager::CONFIG_KEY_CLEANUP_BY_ROWS_LIMIT,
      '#states' => [
        'visible' => [
          ':input[name="' . LoggerDbManager::CONFIG_KEY_CLEANUP_BY_ROWS_ENABLED . '"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['cleanup']['cleanup_now'] = [
      '#type' => 'submit',
      '#name' => 'cleanup_now',
      '#value' => $this->t('Cleanup now'),
    ];
    $form['cleanup']['delete_all'] = [
      '#type' => 'submit',
      '#name' => 'delete_all',
      '#value' => $this->t('Delete all logs'),
      '#attributes' => [
        'class' => ['button--danger'],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    $triggeringElement = $form_state->getTriggeringElement();
    switch ($triggeringElement['#name']) {
      case 'cleanup_now':
        $this->loggerDbManager->cleanupDatabase();
        $this->messenger()->addMessage($this->t('Database logs cleaned up.'));
        break;

      case 'delete_all':
        $this->loggerDbManager->deleteAllLogs();
        $this->messenger()->addMessage($this->t('Database logs deleted.'));
        break;

      case 'reset_logs_view_page':
        $this->loggerDbUtils->resetView();
        $this->messenger()->addMessage($this->t('The <a href="/admin/reports/logs">Logs view page</a> settings restored to defaults.'));
        break;
    }
  }

}
