<?php

namespace Drupal\logger_db\Plugin\views\field;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\Component\Serialization\Yaml;
use Drupal\logger_db\Plugin\views\JsonValuePathFunctionsTrait;
use Drupal\views\Attribute\ViewsField;
use Drupal\views\Plugin\views\field\Standard;
use Drupal\views\ResultRow;

/**
 * Provides a field handler that renders a log severity.
 */
#[ViewsField("logger_db_log_value")]
class LoggerDbLogValueField extends Standard {

  use JsonValuePathFunctionsTrait {
    buildOptionsForm as protected traitBuildOptionsForm;
    defineOptions as protected traitDefineOptions;
  }

  // Display format constants.
  private const DISPLAY_FORMAT_YAML = 'YAML';
  private const DISPLAY_FORMAT_JSON_PRETTY = 'JSON_PRETTY';
  private const DISPLAY_FORMAT_JSON = 'JSON';
  private const DISPLAY_FORMAT_LIST = 'LIST';

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = self::traitDefineOptions();
    $options['value_format'] = [
      'default' => '',
    ];
    $options['display_format'] = [
      'default' => self::DISPLAY_FORMAT_YAML,
    ];
    $options['match_to_value'] = [
      'default' => FALSE,
    ];
    $options['match_value'] = [
      'default' => '',
    ];
    $options['match_use_regexp'] = [
      'default' => FALSE,
    ];
    $options['match_to_value_inverse'] = [
      'default' => FALSE,
    ];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    $form = self::traitBuildOptionsForm($form, $form_state);
    $form['value_format'] = [
      '#type' => 'radios',
      '#title' => $this->t('Value Format'),
      '#description' => $this->t('Allows casting the value to a specific format.'),
      '#options' => [
        self::$valueFormatRaw => $this->t('Raw'),
        self::$valueFormatString => $this->t('String'),
        self::$valueFormatNumeric => $this->t('Numeric'),
      ],
      '#default_value' => $this->options['value_format'],
      '#weight' => -1,
    ];
    $form['display_format'] = [
      '#type' => 'radios',
      '#title' => $this->t('Object Display Format'),
      '#description' => $this->t('Choose how the object value will be displayed.'),
      '#options' => [
        self::DISPLAY_FORMAT_YAML => $this->t('YAML'),
        self::DISPLAY_FORMAT_JSON_PRETTY => $this->t('JSON pretty-printed'),
        self::DISPLAY_FORMAT_JSON => $this->t('JSON'),
        self::DISPLAY_FORMAT_LIST => $this->t('List of values'),
      ],
      '#default_value' => $this->options['display_format'],
      '#states' => [
        'visible' => [
          ':input[name="options[value_format]"]' => ['value' => self::$valueFormatRaw],
        ],
      ],
      '#required' => TRUE,
      '#weight' => -1,
    ];
    $form['match_to_value'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Match to the value'),
      '#default_value' => $this->options['match_to_value'],
    ];
    $form['match_value'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Value to match'),
      '#default_value' => $this->options['match_value'],
      '#states' => [
        'visible' => [
          ':input[name="options[match_to_value]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['match_use_regexp'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use regexp match'),
      '#default_value' => $this->options['match_use_regexp'],
      '#states' => [
        'visible' => [
          ':input[name="options[match_to_value]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['match_to_value_inverse'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Inverse the match result'),
      '#default_value' => $this->options['match_to_value_inverse'],
      '#states' => [
        'visible' => [
          ':input[name="options[match_to_value]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    $this->ensureMyTable();
    if (
      $this->options['value_format'] === self::$valueFormatNumeric
      || $this->options['value_format'] === self::$valueFormatString
    ) {
      $expression = $this->getJsonExpression($this->tableAlias . '.data', $this->options['value_path'], $this->options['value_format']);
    }
    else {
      $expression = $this->getJsonExpression($this->tableAlias . '.data', $this->options['value_path']);
    }
    $params = [];
    if (
      $this->options['group_type']
      && $this->options['group_type'] !== 'group'
    ) {
      $params['function'] = $this->options['group_type'];
    }

    if ($this->options['match_to_value']) {
      $matchTrue = $this->options['match_to_value_inverse'] ? 0 : 1;
      $matchFalse = $this->options['match_to_value_inverse'] ? 1 : 0;
      if ($this->options['match_use_regexp']) {
        $regexpOperator = $this->getDbType() == 'pgsql' ? '~' : 'REGEXP';
        $expression = "(CASE WHEN ($expression) $regexpOperator '{$this->options['match_value']}' THEN $matchTrue ELSE $matchFalse END)";
      }
      else {
        $expression = "(CASE WHEN ($expression) = '{$this->options['match_value']}' THEN $matchTrue ELSE $matchFalse END)";
      }
    }

    /** @var \Drupal\views\Plugin\views\query\Sql $query */
    $query = $this->query;
    $this->field_alias = $query->addField(NULL, $expression, $this->field, $params);
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    $this->view->element['#attached']['library'][] = 'logger_db/fields';
    $value = $this->getValue($values);
    if (!empty($value)) {
      switch ($this->options['value_format']) {
        case self::$valueFormatRaw:
          switch ($this->options['display_format']) {
            case self::DISPLAY_FORMAT_YAML:
              $data = json_decode($value, associative: TRUE);
              $value = Yaml::encode($data);
              $value = Markup::create("<div class='logger-db-pre'>$value</div>");
              break;

            case self::DISPLAY_FORMAT_LIST:
              $data = json_decode($value, associative: TRUE);
              $value = implode(', ', $data);
              $value = Markup::create("<div class='logger-db-pre'>$value</div>");
              break;

            case self::DISPLAY_FORMAT_JSON_PRETTY:
              $data = json_decode($value);
              $value = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
              $value = Markup::create("<div class='logger-db-pre'>$value</div>");
              break;

            case self::DISPLAY_FORMAT_JSON:
            default:
              // Keep the original value without sanitizing.
              break;
          }
          break;

        case self::$valueFormatNumeric:
          $value = $this->sanitizeValue($value);
          break;

        case self::$valueFormatString:
        default:
          // Keep the original value without sanitizing.
          break;
      }
    }
    return $value;
  }

}
