<?php

namespace Drupal\logger_db\Plugin\views\field;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\views\Attribute\ViewsField;
use Drupal\views\Plugin\views\field\Standard;
use Drupal\views\ResultRow;

/**
 * Provides a field handler that renders timestamp values with date formatting.
 *
 * This field handler formats timestamp values (including milliseconds) using
 * Drupal's date formatter service with configurable date formats.
 */
#[ViewsField("logger_db_message")]
class LoggerDbMessage extends Standard {

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['link_to_entry'] = ['default' => FALSE];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    // Call parent to add the main field.
    parent::query();

    if ($this->options['link_to_entry']) {
      // Add additional fields we need (uuid for the link).
      $this->addAdditionalFields(['uuid']);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    $form['link_to_entry'] = [
      '#title' => $this->t('Link to entry'),
      '#description' => $this->t('Make a link to the entry view page. Add the hidden "ID" field to the views to make it work.'),
      '#type' => 'checkbox',
      '#default_value' => !empty($this->options['link_to_entry']),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    $value = $this->getValue($values);
    if ($this->options['link_to_entry']) {
      // If the link to entity option is enabled, wrap the date in a link.
      // Access the UUID value using the field alias created by
      // addAdditionalFields.
      $uuid = $values->{$this->aliases['uuid']} ?? NULL;
      if ($uuid) {
        $url = Url::fromRoute('logger_db.entry', ['uuid' => $uuid]);
        return [
          '#type' => 'link',
          '#title' => $value,
          '#url' => $url,
        ];
      }
    }

    return $value;
  }

}
