const baseUrl = process.env.DRUPAL_TEST_BASE_URL;
const baseUrlPath = new URL(baseUrl).pathname;
const basePath = baseUrlPath === '/' ? '' : baseUrlPath;

let viewId;
let pageId;

function setViewsInfo(newViewId, newPageId) {
  viewId = newViewId;
  pageId = newPageId;
}

function submitModalWindow(browser, buttonSelector) {
  const selector = `.views-ui-dialog .form-actions ${buttonSelector || '.button--primary.ui-widget'} `;
  return browser
    .waitForElementPresent(selector)
    .click(selector)
    .waitForElementNotPresent(selector);
}

function setFieldFormValues(browser, { valuePath, value, operator, values }) {
  if (values) {
    Object.entries(values).forEach(([field, fieldValue]) => {
      browser
        .waitForElementPresent(
          `.views-ui-dialog [data-drupal-selector="edit-options-${field.replace(/_/g, '-')}"]`,
        )
        // Set the custom field value.
        .setValue(
          `.views-ui-dialog [data-drupal-selector="edit-options-${field.replace(/_/g, '-')}"]`,
          fieldValue,
        );
    });
  }
  if (valuePath) {
    browser
      .waitForElementPresent(
        '.views-ui-dialog [data-drupal-selector="edit-options-value-path"]',
      )
      // Set the value path (JSONPath).
      .setValue(
        '.views-ui-dialog [data-drupal-selector="edit-options-value-path"]',
        valuePath,
      );
  }

  if (operator) {
    browser
      // Set the operator (e.g., equals, less than).
      .click(
        `.views-ui-dialog [data-drupal-selector="edit-options-operator"] [value="${operator}"]`,
      );
  }

  if (value) {
    browser.perform(async () => {
      // Set the value.
      if (
        await browser.isPresent({
          selector:
            '.views-ui-dialog [data-drupal-selector="edit-options-value-value"]',
          suppressNotFoundErrors: true,
          timeout: 1,
        })
      ) {
        browser.setValue(
          '.views-ui-dialog [data-drupal-selector="edit-options-value-value"]',
          value,
        );
      } else if (
        await browser.isPresent({
          selector:
            '.views-ui-dialog [data-drupal-selector="edit-options-value"]',
          suppressNotFoundErrors: true,
          timeout: 1,
        })
      ) {
        browser.setValue(
          '.views-ui-dialog [data-drupal-selector="edit-options-value"]',
          value,
        );
      } else {
        throw new Error('Value input field is not found in the dialog.');
      }
    });
  }
}

function addViewsField(
  browser,
  {
    fieldType = 'field',
    pluginName,
    valuePath,
    value,
    operator,
    values,
    perform,
  } = {},
) {
  browser
    // Add the field.
    .click(`#views-add-${fieldType}`)
    .waitForElementPresent('.ui-dialog-content')
    .click(`.views-ui-dialog [value="logger_logs.${pluginName}"]`)
    .thPerformAndWaitForReRender(() => {
      browser.click(
        '.views-ui-dialog .form-actions .button--primary.ui-widget',
      );
    }, '.views-ui-dialog .form-actions .button--primary.ui-widget');

  setFieldFormValues(browser, { valuePath, value, operator, values });
  if (perform) {
    browser.perform(perform(browser));
  }
  submitModalWindow(browser);
}

function editViewsField(
  browser,
  { fieldType, fieldId, valuePath, value, operator, values },
) {
  const editUrl = `${basePath}/admin/structure/views/nojs/handler/${viewId}/${pageId}/${fieldType}/${fieldId}`;
  browser
    // Edit the field by opening the edit dialog.
    .click(`a[href="${editUrl}"]`)
    .waitForElementPresent(
      '.views-ui-dialog .form-actions .button--primary.ui-widget',
    );
  setFieldFormValues(browser, { valuePath, value, operator, values });
  submitModalWindow(browser);
}

function deleteViewsField(browser, { fieldType = 'field', fieldId } = {}) {
  const editUrl = `${basePath}/admin/structure/views/nojs/handler/${viewId}/${pageId}/${fieldType}/${fieldId}`;
  browser
    // Edit the field by opening the edit dialog.
    .click(`a[href="${editUrl}"]`)
    .waitForElementPresent(
      '.views-ui-dialog .form-actions .button--danger.ui-widget',
    );
  submitModalWindow(browser, '.button--danger.ui-widget');
}

async function executeAndGetTable(browser) {
  browser
    // Scroll to bottom to see the results in the screenshot.
    .execute('window.scrollTo(0, document.body.scrollHeight);')
    .waitForElementPresent('[data-drupal-selector="preview-submit"]')
    .thPerformAndWaitForReRender(() => {
      browser.click('[data-drupal-selector="preview-submit"]');
    }, '#views-ui-preview-form table:not([data-drupal-selector])')
    // Scroll to bottom to see the results in the screenshot.
    .execute('window.scrollTo(0, document.body.scrollHeight);');

  let tableData;
  await browser.perform(async () => {
    const query = await browser.getText('.views-query-info pre');
    // eslint-disable-next-line no-console
    console.log('Views query:', query);

    // Check if table exists before trying to read it
    const tableExists = await browser.findElements(
      '#views-ui-preview-form table:not([data-drupal-selector])',
    );

    if (tableExists.length > 0) {
      tableData = await browser.loggerDbGetTableContent(
        '#views-ui-preview-form table:not([data-drupal-selector])',
      );
      // eslint-disable-next-line no-console
      console.log('Table data:', tableData);
    } else {
      tableData = [];
    }
  });
  return tableData;
}

module.exports = {
  basePath,
  setViewsInfo,
  submitModalWindow,
  setFieldFormValues,
  addViewsField,
  editViewsField,
  deleteViewsField,
  executeAndGetTable,
};
