<?php

namespace Drupal\logger_db\Plugin\LoggerTarget;

use Drupal\Core\Database\Connection;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\logger\Attribute\LoggerTarget;
use Drupal\logger\LoggerEntryInterface;
use Drupal\logger\Plugin\LoggerTargetBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Component\Uuid\UuidInterface;

/**
 * Provides a file logger target.
 */
#[LoggerTarget(
  id: 'db',
  label: new TranslatableMarkup('Database'),
  description: new TranslatableMarkup('Writes log to the database. Not recommended for production.'),
  weight: 40
)]
class Db extends LoggerTargetBase implements ContainerFactoryPluginInterface {
  /**
   * The database table name to store logs.
   *
   * @var string
   */
  const DB_TABLE = 'logger_logs';

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $connection;

  /**
   * UUID service.
   *
   * @var \Drupal\Component\Uuid\UuidInterface
   */
  protected UuidInterface $uuid;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    // Store the container and lazily fetch services when needed.
    $instance->connection = $container->get('database');
    $instance->uuid = $container->get('uuid');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function persist(LoggerEntryInterface $entry): void {
    $mt = $entry->getMicrotime();
    $microtimeString = \DateTime::createFromFormat('U.u', sprintf('%.6F', $mt), new \DateTimeZone('UTC'))->format('Y-m-d H:i:s.u');
    $data = $entry->getData();
    $channel = $data['channel'];
    $message = $data['message'] ?? $data['message_raw'] ?? '';
    $this->connection->insert(self::DB_TABLE)
      ->fields([
        'uuid' => $data['uuid'] ?? $this->uuid->generate(),
        'time' => $microtimeString,
        'severity' => $entry->getLogLevel(),
        'channel' => $channel,
        'message' => $message,
        'data' => json_encode($data),
      ])
      ->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function validateTarget(array $target): bool {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultConfiguration(): array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfigForm() {
    return [
      '#markup' => '',
    ];
  }

}
