<?php

namespace Drupal\logger_db\Plugin\views\field;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\TimeZoneFormHelper;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Url;
use Drupal\views\Attribute\ViewsField;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a field handler that renders timestamp values with date formatting.
 *
 * This field handler formats timestamp values (including milliseconds) using
 * Drupal's date formatter service with configurable date formats.
 */
#[ViewsField("logger_db_time")]
class LoggerDbTime extends FieldPluginBase implements ContainerFactoryPluginInterface {

  /**
   * Redefine the variable to set the right type.
   *
   * @var \Drupal\views\Plugin\views\query\Sql
   */
  public $query;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create(...func_get_args());
    $instance->configFactory = $container->get('config.factory');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['date_format'] = ['default' => 'Y-m-d H:i:s'];
    $options['timezone'] = ['default' => ''];
    $options['link_to_entry'] = ['default' => TRUE];
    $options['round_to_seconds'] = ['default' => 0];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    $this->ensureMyTable();
    $params = $this->options['group_type'] != 'group' ? ['function' => $this->options['group_type']] : [];
    $fieldReference = $this->tableAlias . '.' . $this->realField;
    if ($this->options['round_to_seconds'] > 0) {
      $fieldReference = $this->getRoundExpression($fieldReference, $this->options['round_to_seconds']);
    }
    $timeWithTimeZone = $this->getTzConvertExpression($fieldReference);

    $formatString = $this->options['date_format'] ?: 'Y-m-d H:i:s';
    switch ($this->query->getConnection()->getConnectionOptions()['driver']) {
      case 'mysql':
      case 'pgsql':
        $expression = $this->query->getDateFormat($timeWithTimeZone, $formatString, TRUE);
        break;

      case 'sqlite':
        $expression = $this->query->getDateFormat($timeWithTimeZone, $formatString, TRUE);
        // We use time as ISO string, but Drupal's SQLite driver expects
        // unixepoch.
        // @see SqliteDateSql::getDateFormat().
        // Remove the unixepoch modifier manually.
        $expression = str_replace(", 'unixepoch'", '', $expression);
        break;

      default:
        throw new \Exception('Unsupported database driver: ' . $this->query->getConnection()->getConnectionOptions()['driver']);
    }
    // Add the expression as a selected field and capture the alias returned.
    $this->field_alias = $this->query->addField(NULL, $expression, "{$this->tableAlias}_{$this->realField}", $params);

    // When grouping is enabled, group by the SELECT alias instead of the raw
    // expression. Passing the alias lets Views/DB layer generate correct SQL
    // (avoids mangling/quoting of complex expressions).
    if ($this->options['group_type'] == 'group') {
      // Group by the alias returned by addField(). addGroupBy() takes a
      // single clause string; passing the alias avoids Views trying to
      // interpret/quote the raw expression.
      $this->query->addGroupBy($this->field_alias);
    }
    if (
      $this->options['link_to_entry']
      && !$this->view->display_handler->getOption('group_by')
    ) {
      // Add additional fields we need (uuid for the link).
      $this->addAdditionalFields(['uuid']);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    $form['date_format'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Output format'),
      '#description' => $this->t('The date format with millisecond precision. See <a href="https://www.php.net/manual/datetime.format.php#refsect1-datetime.format-parameters" target="_blank">the PHP docs</a> for date formats for example, <code>Y-m-d H:i:s</code>.'),
      '#default_value' => $this->options['date_format'] ?? '',
    ];

    $form['timezone'] = [
      '#type' => 'select',
      '#title' => $this->t('Timezone'),
      '#description' => $this->t('Timezone to be used for date output.'),
      '#options' => ['' => $this->t('- Default site/user timezone -')] + TimeZoneFormHelper::getOptionsListByRegion(),
      '#default_value' => $this->options['timezone'] ?? '',
    ];

    $form['link_to_entry'] = [
      '#title' => $this->t('Link to entry'),
      '#description' => $this->t('Make a link to the entry view page. Does not work with aggregation enabled.'),
      '#type' => 'checkbox',
      '#default_value' => !empty($this->options['link_to_entry']),
    ];
    $form['round_to_seconds'] = [
      '#type' => 'number',
      '#title' => $this->t('Round to seconds'),
      '#description' => $this->t('Rounds the time value to the specific amount of seconds for aggregation. Set 600 to group by 10 minutes, 3600 to group by 1 hour, etc. Set 0 to disable.'),
      '#default_value' => $this->options['round_to_seconds'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    $value = $this->getValue($values);

    if (
      $this->options['link_to_entry']
      && isset($this->aliases['uuid'])
    ) {
      // If the link to entity option is enabled, wrap the date in a link.
      // Access the UUID value using the field alias created by
      // addAdditionalFields.
      $uuid = $values->{$this->aliases['uuid']} ?? NULL;
      if ($uuid) {
        $url = Url::fromRoute('logger_db.entry', ['uuid' => $uuid]);
        return [
          '#type' => 'link',
          '#title' => $value,
          '#url' => $url,
        ];
      }
    }

    return $value;
  }

  /**
   * Generates a timezone convert expression depending on the database type.
   *
   * @param mixed $expression
   *   A field or initial expression.
   *
   * @return string
   *   The timezone convert expression.
   */
  protected function getTzConvertExpression($expression): string {
    switch ($this->query->getConnection()->getConnectionOptions()['driver']) {
      case 'mysql':
        return "CONVERT_TZ($expression, '+00:00', '" . $this->getTimezoneOffset() . "')";

      case 'pgsql':
        return "$expression AT TIME ZONE 'UTC' AT TIME ZONE '" . $this->getTimezoneOffset() . "'";

      case 'sqlite':
        // SQLite does not support named timezones, only fixed offsets.
        return "datetime($expression, '" . $this->getTimezoneOffset() . "')";

      default:
        // Unsupported DB, return the expression unchanged.
        throw new \RuntimeException('Unsupported database driver: ' . $this->query->getConnection()->getConnectionOptions()['driver']);
    }
  }

  /**
   * Generates a rounding expression depending on the database type.
   *
   * @param mixed $value
   *   A field or initial expression.
   * @param int $seconds
   *   The number of seconds to round to.
   *
   * @return string
   *   The rounding expression.
   */
  protected function getRoundExpression($value, $seconds): string {
    switch ($this->query->getConnection()->getConnectionOptions()['driver']) {
      case 'mysql':
      case 'sqlite':
        return "FROM_UNIXTIME(FLOOR(UNIX_TIMESTAMP($value) / $seconds) * $seconds)";

      case 'pgsql':
        return "FROM_UNIXTIME(FLOOR(UNIX_TIMESTAMP($value) / $seconds) * $seconds)";

      default:
        // Unsupported DB, return the expression unchanged.
        throw new \RuntimeException('Unsupported database driver: ' . $this->query->getConnection()->getConnectionOptions()['driver']);
    }
  }

  /**
   * Get the timezone offset string for the configured timezone.
   *
   * @return string
   *   The timezone offset in +HH:MM or -HH:MM format.
   */
  protected function getTimezoneOffset(): string {
    $default_timezone = $this->configFactory->get('system.date')->get('timezone')['default'];
    $tz = new \DateTimeZone($this->options['timezone'] ?: $default_timezone);
    $date = new \DateTime('now', $tz);
    return $date->format('P');
  }

}
