<?php

namespace Drupal\logger_db\Plugin\views\filter;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Datetime\Element\Datetime;
use Drupal\views\Attribute\ViewsFilter;
use Drupal\views\Plugin\views\display\DisplayPluginBase;
use Drupal\views\Plugin\views\query\Sql;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\filter\FilterPluginBase;
use Drupal\views\ViewExecutable;

/**
 * Filters log entries by timestamp range.
 */
#[ViewsFilter('logger_db_time_range')]
class LoggerDbTimeRange extends FilterPluginBase {

  /**
   * {@inheritdoc}
   */
  protected $alwaysMultiple = TRUE;

  /**
   * The date format arguments for render time in queries and summary.
   *
   * @var string
   */
  const SUMMARY_TIME_FORMAT = 'Y-m-d H:i:s';

  /**
   * {@inheritdoc}
   */

  /**
   * {@inheritdoc}
   */
  public function init(ViewExecutable $view, DisplayPluginBase $display, ?array &$options = NULL) {
    parent::init($view, $display, $options);
    // The default value comes as string, but we expect array.
    if ($this->value === "") {
      $this->value = [
        'absolute' => [
          'min' => NULL,
          'max' => NULL,
        ],
        'relative' => [
          'min' => NULL,
          'max' => NULL,
        ],
        'is_relative' => FALSE,
      ];
    }
  }

  /**
   * {@inheritdoc}
   */

  /**
   * {@inheritdoc}
   */
  public function defaultExposeOptions() {
    parent::defaultExposeOptions();
    $this->options['expose']['identifier'] = 'time';
    $this->options['value'] = [
      'absolute' => [
        'min' => NULL,
        'max' => NULL,
      ],
      'relative' => [
        'min' => NULL,
        'max' => NULL,
      ],
      'is_relative' => FALSE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function operators() {
    return [
      'between' => [
        'title' => $this->t('Is between'),
        'short' => $this->t('between'),
        'method' => 'opBetween',
        'values' => 2,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  protected function valueForm(&$form, FormStateInterface $form_state) {
    $element['#tree'] = TRUE;
    $element['#attached']['library'][] = 'logger_db/views';
    $element['#attributes']['class'][] = 'logger-db-views-filter-time-form';
    $element['absolute'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input.logger-db-views-filter-time-is-relative' => ['!checked' => TRUE],
        ],
      ],
    ];
    $element['absolute']['min'] = [
      '#type' => 'datetime',
      '#title' => $this->t('From - To (exact dates)'),
      '#default_value' => !empty($this->value['absolute']['min']) ? new DrupalDateTime($this->value['absolute']['min']) : NULL,
      '#element_validate' => [
        [self::class, 'validateMinTimeValue'],
        [Datetime::class, 'validateDatetime'],
      ],
    ];
    $element['absolute']['max'] = [
      '#type' => 'datetime',
      '#description' => $this->t('Choose the start and end date, time is not required.'),
      '#default_value' => !empty($this->value['absolute']['max']) ? new DrupalDateTime($this->value['absolute']['max']) : NULL,
      '#element_validate' => [
        [self::class, 'validateMaxTimeValue'],
        [Datetime::class, 'validateDatetime'],
      ],
    ];

    $element['relative'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input.logger-db-views-filter-time-is-relative' => ['checked' => TRUE],
        ],
      ],
    ];
    $element['relative']['min'] = [
      '#type' => 'textfield',
      '#size' => 12,
      '#title' => $this->t('From - To (relative)'),
    ];
    $element['relative']['max'] = [
      '#type' => 'textfield',
      '#size' => 12,
      '#description' => $this->t('Example: -7 days 5 min'),
      '#default_value' => !empty($this->value['is_relative']) && !empty($this->value['relative']['max']) ? $this->value['relative']['max'] : '',
    ];

    $element['is_relative'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use relative time'),
      '#default_value' => !empty($this->value['is_relative']),
      '#attributes' => [
        'class' => ['logger-db-views-filter-time-is-relative'],
        // A workaround for the Drupal Core issue
        // https://www.drupal.org/project/drupal/issues/1100170
        'checked' => (int) !empty($this->value['is_relative']),
      ],
    ];

    $element['#attached']['library'][] = 'logger_db/views';
    $form['value'] = $element;
  }

  /**
   * Validate the minimum time value for the filter form element.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param array $complete_form
   *   The complete form structure.
   */
  public static function validateMinTimeValue(&$element, FormStateInterface $form_state, &$complete_form) {
    self::validateTimeValue('00:00:00', $element, $form_state, $complete_form);
  }

  /**
   * Validate the maximum time value for the filter form element.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param array $complete_form
   *   The complete form structure.
   */
  public static function validateMaxTimeValue(&$element, FormStateInterface $form_state, &$complete_form) {
    self::validateTimeValue('23:59:59', $element, $form_state, $complete_form);
  }

  /**
   * Validates the time value for the filter form element.
   *
   * @param string $defaultTime
   *   The default time to use if only date is provided.
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param array $complete_form
   *   The complete form structure.
   */
  public static function validateTimeValue($defaultTime, &$element, FormStateInterface $form_state, &$complete_form) {
    if (
      $element['#value']['date'] !== ''
      && $element['#value']['time'] === ''
    ) {
      $element['#value']['time'] = $defaultTime;
      $value = Datetime::valueCallback($element, $element['#value'], $form_state);
      $form_state->setValueForElement($element, $value);
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function buildValueWrapper(&$form, $wrapper_identifier) {
    parent::buildValueWrapper($form, $wrapper_identifier);
    $form[$wrapper_identifier]['#type'] = 'container';
    $form[$wrapper_identifier]['#attributes']['class'][] = 'views-exposed-form__item';
    $form[$wrapper_identifier]['#attributes']['class'][] = 'logger-db-views-filter-time-wrapper';
  }

  /**
   * {@inheritdoc}
   */

  /**
   * {@inheritdoc}
   */
  protected function valueSubmit($form, FormStateInterface $form_state) {
    $options = $form_state->getValue('options');
    if (isset($options['value']['absolute']['min']) && $options['value']['absolute']['min'] instanceof DrupalDateTime) {
      $options['value']['absolute']['min'] = $options['value']['absolute']['min']->format('c');
    }
    if (isset($options['value']['absolute']['max']) && $options['value']['absolute']['max'] instanceof DrupalDateTime) {
      $options['value']['absolute']['max'] = $options['value']['absolute']['max']->format('c');
    }
    $form_state->setValue('options', $options);
  }

  /**
   * {@inheritdoc}
   */

  /**
   * {@inheritdoc}
   */
  public function adminSummary() {
    $min = $this->t('any');
    $max = $this->t('any');

    // Check if using relative time.
    if (!empty($this->value['is_relative'])) {
      if (!empty($this->value['relative']['min'])) {
        $min = $this->value['relative']['min'];
      }
      if (!empty($this->value['relative']['max'])) {
        $max = $this->value['relative']['max'];
      }
    }
    else {
      // Use absolute time.
      if (!empty($this->value['absolute']['min'])) {
        try {
          $minValue = new DrupalDateTime($this->value['absolute']['min']);
          /**
           * @var \Drupal\Core\Datetime\DrupalDateTime $minValue
           */
          $min = $minValue->format(self::SUMMARY_TIME_FORMAT);
        }
        catch (\Exception) {
          // Leave as 'any' if invalid.
        }
      }
      if (!empty($this->value['absolute']['max'])) {
        try {
          $maxValue = new DrupalDateTime($this->value['absolute']['max']);
          /**
           * @var \Drupal\Core\Datetime\DrupalDateTime $maxValue
           */
          $max = $maxValue->format(self::SUMMARY_TIME_FORMAT);
        }
        catch (\Exception) {
          // Leave as 'any' if invalid.
        }
      }
    }

    $value = $this->t('@min - @max', ['@min' => $min, '@max' => $max]);

    if ($this->isAGroup()) {
      $value .= ', grouped';
    }
    if (!empty($this->options['exposed'])) {
      $value .= ', exposed';
    }

    return $value;
  }

  /**
   * {@inheritdoc}
   */

  /**
   * {@inheritdoc}
   */
  public function query() {
    $this->ensureMyTable();

    // Determine which values to use based on is_relative setting.
    $minValue = NULL;
    $maxValue = NULL;

    if (!empty($this->value['is_relative'])) {
      // Use relative time values.
      if (!empty($this->value['relative']['min'])) {
        try {
          $minValue = new DrupalDateTime($this->value['relative']['min']);
        }
        catch (\Exception) {
          // Invalid relative time, skip.
        }
      }
      if (!empty($this->value['relative']['max'])) {
        try {
          $maxValue = new DrupalDateTime($this->value['relative']['max']);
        }
        catch (\Exception) {
          // Invalid relative time, skip.
        }
      }
    }
    else {
      // Use absolute time values.
      if (!empty($this->value['absolute']['min'])) {
        try {
          $minValue = new DrupalDateTime($this->value['absolute']['min']);
        }
        catch (\Exception) {
          // Invalid absolute time, skip.
        }
      }
      if (!empty($this->value['absolute']['max'])) {
        try {
          $maxValue = new DrupalDateTime($this->value['absolute']['max']);
        }
        catch (\Exception) {
          // Invalid absolute time, skip.
        }
      }
    }

    // If no valid values, return early.
    if ($minValue === NULL && $maxValue === NULL) {
      return;
    }

    $field = $this->ensureMyTable() . '.' . $this->realField;

    $drupalDateTimeFormatArgs = [
      'Y-m-d H:i:s.u',
      ['timezone' => 'UTC'],
    ];

    // Ensure we have a SQL query plugin that supports addWhere().
    if (!$this->query instanceof Sql) {
      return;
    }

    /**
     * @var \Drupal\views\Plugin\views\query\Sql $query
     */
    $query = $this->query;

    if ($minValue !== NULL && $maxValue !== NULL) {
      $query->addWhere(
        $this->options['group'],
        $field,
        [
          $minValue->format(...$drupalDateTimeFormatArgs),
          $maxValue->format(...$drupalDateTimeFormatArgs),
        ],
        'BETWEEN'
      );
    }
    elseif ($minValue !== NULL) {
      /**
       * @var \Drupal\Core\Datetime\DrupalDateTime $minValue
       */
      $query->addWhere(
        $this->options['group'],
        $field,
        $minValue->format(...$drupalDateTimeFormatArgs),
        '>='
      );
    }
    elseif ($maxValue !== NULL) {
      /**
       * @var \Drupal\Core\Datetime\DrupalDateTime $maxValue
       */
      $query->addWhere(
        $this->options['group'],
        $field,
        $maxValue->format(...$drupalDateTimeFormatArgs),
        '<='
      );
    }
  }

}
