<?php

namespace Drupal\logger_db\Controller;

use Drupal\Core\Render\Markup;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Component\Serialization\Yaml;
use Drupal\logger_db\Plugin\LoggerTarget\Db;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for our pages.
 */
class LoggerDbController extends ControllerBase {
  /**
   * A database client.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $connection;

  /**
   * The log entry.
   *
   * @var array
   */
  protected array $entry;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->connection = $container->get('database');
    return $instance;
  }

  /**
   * Displays values of a specific log entry by the entry uuid.
   *
   * @param string $uuid
   *   The entry uuid.
   *
   * @return array
   *   The render array.
   */
  public function entryPage(string $uuid) {
    $build['entry'] = [
      '#type' => 'table',
      '#header' => [
        'field' => $this->t('Field'),
        'value' => $this->t('Value'),
      ],
    ];
    $build['#attached']['library'][] = 'logger_db/fields';
    $entry = $this->getEntry($uuid);
    foreach ($entry as $field => $value) {
      if ($field == 'data') {
        $data = json_decode($value, associative: TRUE);
        foreach ($data as $key => $value) {
          $valueFormatted = is_array($value) ? Yaml::encode($value) : $value;
          $items[] = "<div class='logger-db-item'><dd>$key:</dd><dt><div class='logger-db-pre'>$valueFormatted</div></dt></div>";
        }
        $value = Markup::create(implode('', $items));
      }
      $build['entry']['#rows'][] = [$field, $value];
    }
    return $build;
  }

  /**
   * Gets a log entry from the database by the entry id.
   *
   * @param string $uuid
   *   The entry id.
   *
   * @return array|bool
   *   An array with the entry values.
   */
  public function getEntry(string $uuid) {
    $this->entry ??= $this->connection->select(Db::DB_TABLE)
      ->fields(Db::DB_TABLE, [
        'uuid',
        'time',
        'severity',
        'channel',
        'message',
        'data',
      ])
      ->condition('uuid', $uuid)
      ->execute()
      ->fetchAssoc();
    return $this->entry;
  }

  /**
   * Generates the entry title by the entry id.
   *
   * @param string $uuid
   *   The entry id.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The entry title markup.
   */
  public function getEntryTitle(string $uuid) {
    return $this->t('Log entry at @time', ['@time' => $this->getEntry($uuid)['time']]);
  }

}
